(function captureInviteOnce() {
  const params = new URLSearchParams(window.location.search);
  const attorneyId = params.get("attorneyId");

  if (attorneyId && !sessionStorage.getItem("pendingLawFirmInvite")) {
    sessionStorage.setItem("pendingLawFirmInvite", attorneyId);

    // Clean URL immediately
    const cleanUrl = window.location.origin + window.location.pathname;
    window.history.replaceState({}, document.title, cleanUrl);
  }
})();

firebase.initializeApp(firebaseConfig)

firebase.auth().onAuthStateChanged(async (user) => {
  if (!user) return;

  const db = firebase.firestore();
  const currentUser = user;
  const pendingInvite = sessionStorage.getItem("pendingLawFirmInvite");
  if (!pendingInvite) return;

  const currentPath = window.location.pathname;
  const isProfessionalInfoPage = currentPath.includes("professionalInformation");

  // 🚫 Do NOT run invite logic on profile completion page
  

  console.log("✅ Invite found after login:", pendingInvite);

  // 🚫 EARLY SELF-INVITE CHECK
  if (pendingInvite === user.uid) {
    console.warn("🚫 Self invite blocked");
    sessionStorage.removeItem("pendingLawFirmInvite");
    return;
  }

  const lawFirmSnap = await db.collection("LawFirm").doc(pendingInvite).get();
  if (!lawFirmSnap.exists) {
    sessionStorage.removeItem("pendingLawFirmInvite");
    return;
  }

  // const resume = sessionStorage.getItem("resumeInviteAfterProfile");

  // if (resume && pendingInvite) {
  //   console.log("🔁 Resuming invite after profile completion");
  //   sessionStorage.removeItem("resumeInviteAfterProfile");
  // }

  const firm = lawFirmSnap.data();
  const currentLawFirmId = user.uid;
  const userSnap = await db.collection("LawFirm").doc(currentLawFirmId).get();
  const userData = userSnap.exists ? userSnap.data() : { Name: "Attorney" };

  // 🚧 CHECK PROFILE COMPLETENESS BEFORE SHOWING MODAL
  if (!userData.professionalName) {
    console.log("🚧 Profile incomplete, redirecting...");
    Swal.fire({
      icon: "info",
      title: "Complete Your Profile",
      text: "Please complete your professional information before joining a firm.",
      confirmButtonText: "Continue",
    }).then((result) => {
      if (result.isConfirmed) {
        sessionStorage.setItem("resumeInviteAfterProfile", "true");
  
        // ✅ SAFE redirect
        if (isProfessionalInfoPage) {
          console.log("⛔ Invite flow paused on professionalInformation page");
          return;
        }else{
          window.location.href = "/dueproupdated/site/professionalInformation";
        }
      }
    });
    return;
  }

   // 🚫 NEW FEATURE 1: CHECK IF USER IS A FIRM HEAD (has attorneys attached)
  if (
    Array.isArray(userData.attachedLawfirmAttorneys) &&
    userData.attachedLawfirmAttorneys.length > 0
  ) {
    console.warn("🚫 User is a firm head and cannot join another firm");
    sessionStorage.removeItem("pendingLawFirmInvite");
    Swal.fire({
      icon: "warning",
      title: "Cannot Join Firm",
      text: "You are currently managing your own firm with attached attorneys. First delete your subordinates then you can join another firm.",
      confirmButtonText: "OK",
    });
    return;
  }

  // 🚫 CHECK IF ALREADY A MEMBER
  const alreadyMember =
    Array.isArray(firm.attachedLawfirmAttorneys) &&
    firm.attachedLawfirmAttorneys.includes(user.uid);

  if (alreadyMember) {
    console.warn("🚫 Already a member of this firm");
    sessionStorage.removeItem("pendingLawFirmInvite");
    Swal.fire("Info", "You are already part of this firm.", "info");
    return;
  }

  if (
    userData.attachedWithLawfirmId &&
    userData.attachedWithLawfirmId !== pendingInvite
  ) {
    console.log("⚠️ User attached to another firm, asking for confirmation");
    
    // Fetch the current firm head details to show attorney name
    const currentFirmHeadSnap = await db
      .collection("LawFirm")
      .doc(userData.attachedWithLawfirmId)
      .get();
    
    if (!currentFirmHeadSnap.exists) {
      console.warn("⚠️ Current firm head not found");
      sessionStorage.removeItem("pendingLawFirmInvite");
      Swal.fire("Error", "Current firm information not found.", "error");
      return;
    }

    const currentFirmHead = currentFirmHeadSnap.data();
    const currentAttorneyName = currentFirmHead.professionalName || currentFirmHead.Name || "your current attorney";
    const newAttorneyName = firm.professionalName || firm.Name || "the new attorney";

    const confirmResult = await Swal.fire({
      icon: "question",
      title: "Leave Current Firm?",
      html: `You are currently working with <strong>${currentAttorneyName}</strong>.<br><br>Are you sure you want to leave <strong>${currentAttorneyName}</strong> and join <strong>${newAttorneyName}</strong>?`,
      showCancelButton: true,
      confirmButtonText: "Yes, Switch Firms",
      cancelButtonText: "Cancel",
      confirmButtonColor: "#d33",
      cancelButtonColor: "#3085d6",
    });

    if (!confirmResult.isConfirmed) {
      console.log("❌ User cancelled firm switch");
      sessionStorage.removeItem("pendingLawFirmInvite");
      return;
    }

    console.log("✅ User confirmed firm switch");
    // Continue to show the invite modal
  }

  // Populate modal
  document.getElementById("inviteText").textContent =
    `Hi ${userData.professionalName}, ${firm.professionalName} invited you to join ${firm.lawfirmName} on DuePro, the legal platform that connects attorneys and clients in real time.`;

  document.getElementById("inviteFirmName").textContent = firm.lawfirmName;
  document.getElementById("inviteFirmImage").src =
    firm.imageUrl || "../assets/images/th.jpeg";
  document.getElementById("inviteFirmLocation").innerHTML =
    `<i class="fa-solid fa-location-dot"></i> ${firm.address || ""}`;
  document.getElementById("inviteFirmPhone").innerHTML =
    `<i class="fa-solid fa-phone"></i> ${firm.phone || ""}`;
  document.getElementById("inviteFirmWebsite").innerHTML =
    `<i class="fa-solid fa-globe"></i> <a href="${firm.website || "#"}" target="_blank">${firm.website || ""}</a>`;

  const inviteModalEl = document.getElementById("acceptInviteModal");
  const inviteModal = new bootstrap.Modal(inviteModalEl);
  inviteModal.show();

  document.getElementById("acceptInviteBtn").onclick = async () => {
    try {
      Swal.fire({
        title: "Processing",
        allowOutsideClick: false,
        allowEscapeKey: false,
        didOpen: () => {
          Swal.showLoading();
        },
      });
      const pendingInvite = sessionStorage.getItem("pendingLawFirmInvite");
      const user = firebase.auth().currentUser;
      if (!pendingInvite || !user) return;

      const db = firebase.firestore();

      const firmRef = db.collection("LawFirm").doc(pendingInvite);
      const userRef = db.collection("LawFirm").doc(user.uid);

      // ❌ Self-invite protection
      if (pendingInvite === user.uid) {
        Swal.fire("Error", "You cannot join your own firm.", "error");
        sessionStorage.removeItem("pendingLawFirmInvite");
        return;
      }

      // Fetch fresh docs to prevent race conditions
      const [firmSnap, userSnap] = await Promise.all([
        firmRef.get(),
        userRef.get(),
      ]);

      if (!firmSnap.exists || !userSnap.exists) {
        Swal.fire("Error", "Invalid firm data.", "error");
        sessionStorage.removeItem("pendingLawFirmInvite");
        return;
      }

      const firm = firmSnap.data();
      const userFirm = userSnap.data();


      // 🚫 DUPLICATE CHECK: Already in THIS firm
      if (
        Array.isArray(firm.attachedLawfirmAttorneys) &&
        firm.attachedLawfirmAttorneys.includes(user.uid)
      ) {
        Swal.fire("Info", "You are already part of this firm.", "info");
        sessionStorage.removeItem("pendingLawFirmInvite");
        return;
      }

      // 🚫 DUPLICATE CHECK: Already in ANOTHER firm
      // if (
      //   userFirm.attachedWithLawfirmId &&
      //   userFirm.attachedWithLawfirmId !== pendingInvite
      // ) {
      //   Swal.fire({
      //     icon: "warning",
      //     title: "Already in a Firm",
      //     text: `You are already a member of ${userFirm.attachedWithLawfirmName || "another firm"}. Please leave that firm before joining a new one.`,
      //   });
      //   sessionStorage.removeItem("pendingLawFirmInvite");
      //   return;
      // }

      // 🚫 DUPLICATE CHECK: Firm array contains user
      if (userFirm.attachedWithLawfirmId === pendingInvite) {
        Swal.fire("Info", "You are already part of this firm.", "info");
        sessionStorage.removeItem("pendingLawFirmInvite");
        return;
      }

       // ✅ STEP 1: Remove from old firm if attached to one
       if (userFirm.attachedWithLawfirmId) {
        const oldFirmRef = db.collection("LawFirm").doc(userFirm.attachedWithLawfirmId);
        const oldFirmSnap = await oldFirmRef.get();
        
        if (oldFirmSnap.exists) {
          const oldFirmData = oldFirmSnap.data();
          const updatedAttorneys = Array.isArray(oldFirmData.attachedLawfirmAttorneys)
            ? oldFirmData.attachedLawfirmAttorneys.filter(id => id !== user.uid)
            : [];
          
          await oldFirmRef.update({
            attachedLawfirmAttorneys: updatedAttorneys
          });
          console.log("✅ Removed from old firm");
        }
      }

      // ✅ STEP 2: Add to new firm's attachedLawfirmAttorneys array
      const currentAttorneys = Array.isArray(firm.attachedLawfirmAttorneys)
        ? firm.attachedLawfirmAttorneys
        : [];
      
      if (!currentAttorneys.includes(user.uid)) {
        await firmRef.update({
          attachedLawfirmAttorneys: firebase.firestore.FieldValue.arrayUnion(user.uid)
        });
        console.log("✅ Added to new firm's attorneys array");
      }

      // ✅ STEP 3: Update user's attachedWithLawfirmId and firm name
      await userRef.update({
        attachedWithLawfirmId: pendingInvite,
        attachedWithLawfirmName: firm.lawfirmName || firm.professionalName || ""
      });
      console.log("✅ Updated user's firm attachment");

      inviteModal.hide();
      sessionStorage.removeItem("pendingLawFirmInvite");
      
      Swal.fire({
        icon: "success",
        title: "Success!",
        text: `You have successfully joined ${firm.lawfirmName || firm.professionalName}!`,
      }).then(() => {
        // Optional: Reload page to reflect changes
        window.location.reload();
      });

    } catch (err) {
      Swal.fire("Info", "Error loading attorneys: ${err.message}", "info");
      sessionStorage.removeItem("pendingLawFirmInvite");
    }
  };

  // ❌ REJECT
  document.getElementById("rejectInviteBtn").onclick = () => {
    sessionStorage.removeItem("pendingLawFirmInvite");
    inviteModal.hide();
    Swal.fire("Rejected", "Invite declined.", "info");
  };
});