<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Order_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('common');
    }

    /**
     * Get all orders from Firebase Orders collection
     */
    public function get_all_orders()
    {
        return $this->common->readdatafromcollectionwhereclause('Orders');
    }

    /**
     * Get orders filtered by date range
     * @param string $start_date Start date in Y-m-d format
     * @param string $end_date End date in Y-m-d format
     */
    public function get_orders_by_date_range($start_date, $end_date)
    {
        $start_timestamp = strtotime($start_date) * 1000; // Convert to milliseconds
        $end_timestamp = strtotime($end_date . ' 23:59:59') * 1000; // End of day
        
        return $this->common->readdatafromcollectionbetween(
            'Orders', 
            'date', 
            '>=', 
            $start_timestamp, 
            'date', 
            '<=', 
            $end_timestamp
        );
    }

    /**
     * Get today's orders
     */
    public function get_today_orders()
    {
        $today = date('Y-m-d');
        return $this->get_orders_by_date_range($today, $today);
    }

    /**
     * Get orders by specific date
     * @param string $date Date in Y-m-d format
     */
    public function get_orders_by_date($date)
    {
        return $this->get_orders_by_date_range($date, $date);
    }

    /**
     * Get order count by date range
     * @param string $start_date Start date in Y-m-d format
     * @param string $end_date End date in Y-m-d format
     */
    public function get_order_count_by_date_range($start_date, $end_date)
    {
        $start_timestamp = strtotime($start_date) * 1000;
        $end_timestamp = strtotime($end_date . ' 23:59:59') * 1000;
        
        return $this->common->readnumberofdocumentsincollectionbetween(
            'Orders', 
            'date', 
            '>=', 
            $start_timestamp, 
            'date', 
            '<=', 
            $end_timestamp
        );
    }

    /**
     * Get today's order count
     */
    public function get_today_order_count()
    {
        $today = date('Y-m-d');
        return $this->get_order_count_by_date_range($today, $today);
    }

    /**
     * Get order details by ID
     * @param string $order_id Order ID
     */
    public function get_order_by_id($order_id)
    {
        return $this->common->readdatadocument('Orders', $order_id);
    }

    /**
     * Update order status
     * @param string $order_id Order ID
     * @param string $status New status
     */
    public function update_order_status($order_id, $status)
    {
        $data = array('status' => $status);
        return $this->common->updatedatamodel('Orders', $order_id, $data);
    }

    /**
     * Delete order
     * @param string $order_id Order ID
     */
    public function delete_order($order_id)
    {
        return $this->common->deletedatamodel('Orders', $order_id);
    }

    /**
     * Get orders with status filter
     * @param string $status Order status
     */
    public function get_orders_by_status($status)
    {
        return $this->common->readdatafromcollectionwhereclause('Orders', 'status', '=', $status);
    }

    /**
     * Get orders with multiple filters (date range + status)
     * @param string $start_date Start date in Y-m-d format
     * @param string $end_date End date in Y-m-d format
     * @param string $status Order status (optional)
     */
    public function get_orders_with_filters($start_date, $end_date, $status = null)
    {
        $start_timestamp = strtotime($start_date) * 1000;
        $end_timestamp = strtotime($end_date . ' 23:59:59') * 1000;
        
        if ($status) {
            return $this->common->readdatafromcollectionbetweenwhere(
                'Orders', 
                'date', 
                '>=', 
                $start_timestamp, 
                'date', 
                '<=', 
                $end_timestamp,
                'status',
                '=',
                $status
            );
        } else {
            return $this->common->readdatafromcollectionbetween(
                'Orders', 
                'date', 
                '>=', 
                $start_timestamp, 
                'date', 
                '<=', 
                $end_timestamp
            );
        }
    }
}
