<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Mail\RideReceiptMail;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Services\ReceiptService;
use App\Services\FirebaseService;
use App\Services\GmailService;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Mail;

class GeneratePdfController extends Controller
{
    protected ReceiptService $receiptService;
    protected FirebaseService $firebaseService;
    protected GmailService $gmailService;

    public function __construct(ReceiptService $receiptService, FirebaseService $firebaseService, GmailService $gmailService)
    {
        $this->receiptService = $receiptService;
        $this->firebaseService = $firebaseService;
        $this->gmailService = $gmailService;
    }
    public function generateAndEmailReceipt(Request $request, string $rideId)
    {
        // Step 1: Get ride data
        $rideData = $this->firebaseService->getRide($rideId);
        if (!$rideData) {
            abort(404, "Ride not found");
        }

        // Step 2: Normalize
        $receipt = $this->receiptService->normalize($rideData);

        if (empty($receipt['mapImageUrl'])) {
            throw new \Exception('Map image URL missing');
        }

        // Step 3: Convert map image to Base64
        $mapUrl = $receipt['mapImageUrl'];
        $imageData = @file_get_contents($mapUrl);

        if ($imageData === false) {
            throw new \Exception('Failed to load map image');
        }

        $receipt['mapImageBase64'] =
            'data:image/png;base64,' . base64_encode($imageData);

        // Step 4: Generate PDF
        $pdf = Pdf::loadView('receipt', ['receipt' => $receipt]);
        $pdfContent = $pdf->output();

        // Step 5: Prepare email body
        $htmlBody = view('receipt', ['receipt' => $receipt])->render();

        $attachments = [
            [
                'name' => 'receipt.pdf',
                'content' => $pdfContent,
                'mime' => 'application/pdf',
            ],
        ];

        // Step 6: Send email
        try {
            $result = $this->gmailService->sendEmail(
                $receipt['customerEmail'],
                'Your Ride Receipt',
                $htmlBody,
                $attachments
            );

            return response()->json([
                'success' => true,
                'message' => 'PDF emailed successfully',
                'messageId' => $result->getId(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error sending email: ' . $e->getMessage(),
            ], 500);
        }
    }
}
