<?php

namespace App\Services;

use Kreait\Firebase\Factory;

class FirebaseService
{
    protected $firestore;

    /**
     * Initialize Firebase Firestore
     */
    public function __construct()
    {
        $serviceAccountPath = config_path('kataaru-firebase.json');

        $factory = (new Factory)
            ->withServiceAccount($serviceAccountPath);

        // $this->firestore is now the Firestore wrapper from Kreait
        $this->firestore = $factory->createFirestore()->database();
    }

    /**
     * Get a ride document by its ID, and include the user's email
     *
     * @param string $rideId
     * @return array|null
     */
    public function getRide(string $rideId): ?array
    {
        $document = $this->firestore->collection('Bookings')->document($rideId)->snapshot();

        if (!$document->exists()) {
            return null;
        }

        $rideData = $document->data();

        if (isset($rideData['userId'])) {
            $rideData['userEmail'] = $this->getUserEmail($rideData['userId']);
        } else {
            $rideData['userEmail'] = null;
        }
        // Fetch vehicle details
        if (isset($rideData['driverId'])) {
            $rideData['vehicleDetails'] = $this->getVehicleByDriver($rideData['driverId']);
        } else {
            $rideData['vehicleDetails'] = null;
        }

        if (isset($rideData['driverId'])) {
            $rideData['driverData'] = $this->getDriverData($rideData['driverId']);
        } else {
            $rideData['driverData'] = null;
        }
        return $rideData;
    }

    /**
     * Get a user's email by userId
     *
     * @param string $userId
     * @return string|null
     */
    public function getUserEmail(string $userId): ?string
    {
        $document = $this->firestore->collection('Users')->document($userId)->snapshot();

        if (!$document->exists()) {
            return null;
        }

        $data = $document->data();
        return $data['email'] ?? null;
    }
    /**
     * Get vehicle details by driverId
     *
     * @param string $driverId
     * @return array|null
     */
    public function getVehicleByDriver(string $driverId): ?array
    {
        // Assuming Vehicles collection has driverId as document ID
        $document = $this->firestore->collection('Vehicles')->document($driverId)->snapshot();

        if (!$document->exists()) {
            return null;
        }

        return $document->data();
    }
    public function getDriverData(string $driverId): ?array
    {
        // Assuming Users collection has driverId as document ID
        $document = $this->firestore->collection('Users')->document($driverId)->snapshot();

        if (!$document->exists()) {
            return null;
        }

        $data = $document->data();
        $firstName = $data['firstName'] ?? null;
        $lastName = $data['lastName'] ?? null;
        $fullName = trim("$firstName $lastName");
        $returnData =
            [
                'driverName' => $fullName ?? "driver",
                'driverEmail' => $data['email'] ?? null,
            ];
        return $returnData;
    }
}
