<?php

namespace App\Services;

use Google\Client;
use Google\Service\Gmail;
use Google\Service\Gmail\Message;
use Illuminate\Support\Facades\Cache;

class GmailService
{
    protected Client $client;
    protected Gmail $service;

    public function __construct()
    {
        $this->client = new Client();
        $this->client->setClientId(config('services.google.client_id'));
        $this->client->setClientSecret(config('services.google.client_secret'));
        $this->client->setRedirectUri(config('services.google.redirect'));
        $this->client->addScope(Gmail::GMAIL_SEND);
        $this->client->setAccessType('offline');

        // DO NOT force consent
         //$this->client->setPrompt('consent'); ❌ REMOVED

        $token = Cache::get('gmail_access_token');
        if ($token) {
            $this->client->setAccessToken($token);
        }

        // ✅ NO refresh here
        $this->service = new Gmail($this->client);
    }


    /**
     * Refresh access token using refresh token
     */
protected function refreshToken(): void
{
    $refreshToken = Cache::get('gmail_refresh_token');

    if (!$refreshToken) {
        throw new \Exception('Gmail authorization expired. Re-auth required.');
    }

    $newToken = $this->client->fetchAccessTokenWithRefreshToken($refreshToken);

    if (isset($newToken['error'])) {
        Cache::forget('gmail_refresh_token');
        throw new \Exception('Gmail re-authentication required.');
    }

    //  Handle refresh-token rotation
    if (!empty($newToken['refresh_token'])) {
        Cache::forever('gmail_refresh_token', $newToken['refresh_token']);
    }

    $this->client->setAccessToken($newToken);

    Cache::put(
        'gmail_access_token',
        $newToken,
        $newToken['expires_in'] ?? 3600
    );
}


    /**
     * Send email via Gmail API
     */
    public function sendEmail(string $to, string $subject, string $body, array $attachments = [])
    {
        if ($this->client->isAccessTokenExpired()) {
            $this->refreshToken();
        }

        $message = $this->createMessage($to, $subject, $body, $attachments);

        return $this->service->users_messages->send('me', $message);
    }
public function authenticate(string $code): array
{
    $token = $this->client->fetchAccessTokenWithAuthCode($code);

    if (isset($token['error'])) {
        throw new \Exception($token['error_description'] ?? 'Token fetch failed');
    }

    Cache::put(
        'gmail_access_token',
        $token,
        $token['expires_in'] ?? 3600
    );

    if (!empty($token['refresh_token'])) {
        Cache::forever('gmail_refresh_token', $token['refresh_token']);
    }

    $this->client->setAccessToken($token);

    return $token;
}
    /**
     * Build Gmail message
     */
    protected function createMessage(string $to, string $subject, string $body, array $attachments = []): Message
    {
        $fromEmail = config('mail.from.address');
        $fromName  = config('mail.from.name');

        $boundary = uniqid('boundary_');

        $rawMessage  = "From: {$fromName} <{$fromEmail}>\r\n";
        $rawMessage .= "To: {$to}\r\n";
        $rawMessage .= "Subject: {$subject}\r\n";
        $rawMessage .= "MIME-Version: 1.0\r\n";
        $rawMessage .= "Content-Type: multipart/mixed; boundary=\"{$boundary}\"\r\n\r\n";

        // Body
        $rawMessage .= "--{$boundary}\r\n";
        $rawMessage .= "Content-Type: text/html; charset=UTF-8\r\n";
        $rawMessage .= "Content-Transfer-Encoding: quoted-printable\r\n\r\n";
        $rawMessage .= quoted_printable_encode($body) . "\r\n\r\n";

        // Attachments
        foreach ($attachments as $file) {
            $rawMessage .= "--{$boundary}\r\n";
            $rawMessage .= "Content-Type: {$file['mime']}; name=\"{$file['name']}\"\r\n";
            $rawMessage .= "Content-Disposition: attachment; filename=\"{$file['name']}\"\r\n";
            $rawMessage .= "Content-Transfer-Encoding: base64\r\n\r\n";
            $rawMessage .= chunk_split(base64_encode($file['content'])) . "\r\n";
        }

        $rawMessage .= "--{$boundary}--";

        $message = new Message();
        $message->setRaw($this->base64UrlEncode($rawMessage));

        return $message;
    }

    /**
     * Gmail requires base64url encoding
     */
    protected function base64UrlEncode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    /**
     * Get OAuth URL
     */
    public function getAuthUrl(): string
    {
//uncommnet when want to add or get refresh token
        //$this->client->setPrompt('consent select_account');
        return $this->client->createAuthUrl();
    }
}

