<?php

namespace App\Services;

class ReceiptService
{
    protected MapRouteService $mapRouteService;
    protected StaticMapService $staticMapService;


    public function __construct(MapRouteService $mapRouteService, StaticMapService $staticMapService)
    {
        $this->mapRouteService = $mapRouteService;
        $this->staticMapService = $staticMapService;
    }
    /**
     * Normalize raw ride data into receipt-friendly format
     *
     * @param array $rideData
     * @return array
     */
    public function normalize(array $rideData): array
    {
        // Format pickup
        $pickup = $rideData['pickUp']['address'] ?? 'N/A';

        // Process destinations / stops
        $stops = [];
        $finalDestination = null;

        if (!empty($rideData['destinations'])) {
            if (count($rideData['destinations']) === 1) {
                $finalDestination = $rideData['destinations'][0]['address'];
            } else {
                foreach ($rideData['destinations'] as $index => $dest) {
                    if ($index === count($rideData['destinations']) - 1) {
                        $finalDestination = $dest['address'];
                    } else {
                        $stops[] = $dest['address'];
                    }
                }
            }
        }

        // Format date
        $date = isset($rideData['pickupDate']) ? date('M d, Y H:i', $rideData['pickupDate'] / 1000) : 'N/A';

        // Distance & time
        $distance = $rideData['distance'] ?? 'N/A';
        $time = $rideData['time'] ?? 'N/A';

        // Payment
        $amount = isset($rideData['rideAmount'], $rideData['currency'])
            ? number_format((float)$rideData['rideAmount'], 2) . ' ' . $rideData['currency']
            : 'N/A';

        // Vehicle details
        $vehicle = $rideData['vehicleDetails'] ?? [];

        // Generate route points & map image
        $routePoints = [];
        $mapImageUrl = '';
        if (!empty($rideData['directionsResponse'])) {
            $routePoints = $this->mapRouteService->getFullRoute($rideData['directionsResponse']);
            $mapImageUrl = $this->staticMapService->generateMapImage($routePoints);
        }


        // Build normalized receipt
        $receipt = [
            'rideId' => $rideData['id'] ?? 'N/A',
            'date' => $date,
            'customerName' => $rideData['customerName'] ?? 'N/A',
            'customerEmail' => $rideData['userEmail'] ?? 'N/A',
            'driverName' => $rideData['driverData']['driverName'] ?? 'N/A',
            'driverEmail' => $rideData['driverData']['driverEmail'] ?? 'N/A',
            'vehicleType' => $rideData['vehicleType'] ?? $vehicle['carType'] ?? 'N/A',
            'vehicleMake' => $vehicle['make'] ?? 'N/A',
            'vehicleModel' => $vehicle['model'] ?? 'N/A',
            'vehicleYear' => $vehicle['year'] ?? 'N/A',
            'vehicleTagNumber' => $vehicle['tagNumber'] ?? 'N/A',
            'pickup' => $pickup,
            'stops' => $stops,
            'destination' => $finalDestination,
            'distance' => $distance,
            'time' => $time,
            'paymentType' => $rideData['paymentType'] ?? 'N/A',
            'amount' => $amount,
            // 'routePoints' => $routePoints,
            'mapImageUrl' => $mapImageUrl,



        ];

        return $receipt;
    }
}
