<?php
ob_start();

$formatDuration = static function ($seconds) {
    $seconds = max(0, (int)$seconds);
    $minutes = floor($seconds / 60);
    $remainingSeconds = $seconds % 60;
    return sprintf('%d:%02d', $minutes, $remainingSeconds);
};

$formatPrice = static function ($amount) {
    if ($amount === null || $amount === '' || !is_numeric($amount)) {
        return 'N/A';
    }
    return '$' . number_format((float)$amount, 2);
};
?>

<div class="d-flex flex-column flex-md-row justify-content-between align-items-start align-items-md-center mb-3">
    <h3 class="fh2 fmon mb-2 mb-md-0">Workout Detail</h3>

    <div class="d-flex flex-wrap gap-2">
        <a href="<?= base_url('dashboard/workout-plans') ?>" class="btn btn-outline-dark btn-sm">
            ← Back to List
        </a>
        <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#inviteModal">
            Invite to Workout
        </button>
        <button class="btn btn-outline-success btn-sm" onclick="openAddSubWorkoutModal('<?= $plan['id'] ?>')">
            + Add Sub-Workout
        </button>
        <button class="btn btn-outline-primary btn-sm" onclick="openEditModal('<?= $plan['id'] ?>')">
            Edit
        </button>
        <button class="btn btn-outline-danger btn-sm" onclick="confirmDeletePlan('<?= $plan['id'] ?>')">
            Delete
        </button>
    </div>
</div>

<!-- Pricing Summary -->
<div class="row g-3 mb-4">
    <div class="col-md-6">
        <div class="card h-100 shadow-sm">
            <div class="card-body d-flex justify-content-between align-items-center">
                <div>
                    <p class="mb-1 text-muted fw-semibold text-uppercase small">Monthly Price</p>
                    <h5 class="mb-0"><?= htmlspecialchars($formatPrice($plan['monthlyPrice'] ?? null)) ?></h5>
                </div>
                <span class="badge bg-primary-subtle text-primary px-3 py-2">Monthly</span>
            </div>
        </div>
    </div>
    <div class="col-md-6">
        <div class="card h-100 shadow-sm">
            <div class="card-body d-flex justify-content-between align-items-center">
                <div>
                    <p class="mb-1 text-muted fw-semibold text-uppercase small">Annual Price</p>
                    <h5 class="mb-0"><?= htmlspecialchars($formatPrice($plan['annualPrice'] ?? null)) ?></h5>
                </div>
                <span class="badge bg-success-subtle text-success px-3 py-2">Yearly</span>
            </div>
        </div>
    </div>
</div>


<!-- ========================== ADD SUB WORKOUT MODAL ========================== -->
<div class="modal fade" id="addSubWorkoutModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content" style="border-radius:16px; border:1px solid #dee2e6; box-shadow:0 8px 24px rgba(0,0,0,0.15);">
            <form id="addSubWorkoutForm" enctype="multipart/form-data">
                <div class="modal-header bg-light">
                    <h5 class="modal-title">Add Sub-Workout</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>

                <div class="modal-body" style="padding:1.5rem;">
                    <input type="hidden" name="planId" id="parentPlanId" value="<?= htmlspecialchars($plan['id']) ?>">
                    <input type="hidden" name="duration" id="subWorkoutDuration" value="0">

                    <!-- Title -->
                    <div class="mb-3 row">
                        <div class="col-md-6">
                            <label for="workoutTitle" class="form-label fw-semibold">Workout Title</label>
                            <input type="text" class="form-control" id="workoutTitle" name="title" maxlength="255" required>
                            <small id="workoutTitleError" class="text-danger"></small>
                        </div>
                    </div>

                    <!-- Description + Video -->
                    <div class="mb-3 row">
                        <div class="col-md-6">
                            <label for="workoutDescription" class="form-label fw-semibold">Workout Details</label>
                            <textarea class="form-control" id="workoutDescription" name="description" rows="6" placeholder="Details..." maxlength="2000"></textarea>
                            <small id="workoutDescriptionError" class="text-danger"></small>
                        </div>

                        <div class="mb-3 col-md-6">
                            <label for="video" class="form-label fw-semibold">Movements (Video)</label>
                            <label style="height: 88%; border:2px dashed #ccc; border-radius:8px; cursor:pointer;" for="video" class="d-flex justify-content-center align-items-center">
                                <input type="file" id="video" name="video" style="display:none;" accept="video/*">
                                <div class="d-flex justify-content-center flex-column align-items-center">
                                    <img src="<?= base_url('dashboard-assets/images/megaIcons/upload.png') ?>" alt="" style="width:40px; opacity:0.7;">
                                    <p class="text-center mt-2 mb-0 text-muted">Tap to Upload Video</p>
                                </div>
                            </label>
                            <small class="text-muted d-block mt-1">Accepted formats: MP4, MOV, AVI — Max size: 50MB</small>
                            <small id="videoError" class="text-danger"></small>
                            <div class="mt-2" id="videoPreview"></div>
                        </div>
                    </div>

                    <!-- Multiple Images Upload -->
                    <div class="row">
                        <div class="mb-3 col-md-6">
                            <label for="subImages" class="form-label fw-semibold">Sub Workout Images</label>
                            <input type="file" class="form-control" id="subImages" name="images[]" accept="image/*" multiple>
                            <small class="text-muted">Accepted formats: JPG, PNG, GIF — Max size: 5MB per image. You can select multiple images.</small>
                            <small id="subImagesError" class="text-danger"></small>

                            <!-- Multiple Images Preview Container -->
                            <div class="mt-3" id="multipleImagesPreview">
                                <div class="d-flex flex-wrap gap-2" id="selectedImagesContainer"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">Add Sub-Workout</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- ========================== EDIT SUB WORKOUT MODAL ========================== -->
<div class="modal fade" id="editSubWorkoutModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content" style="border-radius:16px; border:1px solid #dee2e6; box-shadow:0 8px 24px rgba(0,0,0,0.15);">
            <form id="editSubWorkoutForm" enctype="multipart/form-data">
                <div class="modal-header bg-light">
                    <h5 class="modal-title">Edit Sub-Workout</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>

                <div class="modal-body" style="padding:1.5rem;">
                    <input type="hidden" id="editPlanIdSub" name="planId">
                    <input type="hidden" id="editSubId" name="subId">
                    <input type="hidden" id="existingImagesData" name="existingImages">
                    <input type="hidden" id="editSubDuration" name="duration" value="0">

                    <!-- Title -->
                    <div class="mb-3 row">
                        <div class="col-md-6">
                            <label for="editSubTitle" class="form-label fw-semibold">Workout Title</label>
                            <input type="text" class="form-control" id="editSubTitle" name="title" maxlength="255" required>
                        </div>
                    </div>

                    <!-- Description + Video -->
                    <div class="mb-3 row">
                        <div class="col-md-6">
                            <label for="editSubDescription" class="form-label fw-semibold">Workout Details</label>
                            <textarea class="form-control" id="editSubDescription" name="description" rows="6" placeholder="Details..." maxlength="2000"></textarea>
                        </div>

                        <div class="col-md-6">
                            <label for="editSubVideo" class="form-label fw-semibold">Movements (Video)</label>
                            <label style="height:88%; border:2px dashed #ccc; border-radius:8px; cursor:pointer;" for="editSubVideo" class="d-flex justify-content-center align-items-center">
                                <input type="file" id="editSubVideo" name="video" style="display:none;" accept="video/*">
                                <div class="d-flex justify-content-center flex-column align-items-center">
                                    <img src="<?= base_url('dashboard-assets/images/megaIcons/upload.png') ?>" alt="" style="width:40px; opacity:0.7;">
                                    <p class="text-center mt-2 mb-0 text-muted">Tap to Replace Video</p>
                                </div>
                            </label>
                            <small class="text-muted d-block mt-1">Accepted: MP4, MOV, AVI — Max: 50MB</small>
                            <div class="mt-2" id="editVideoPreview"></div>
                        </div>
                    </div>

                    <!-- Existing Images Section -->
                    <div class="row">
                        <div class="mb-3 col-12">
                            <label class="form-label fw-semibold">Current Images</label>
                            <div class="alert alert-info py-2">
                                <small><i class="bi bi-info-circle me-1"></i> Click the <i class="bi bi-trash text-danger"></i> icon to remove images. Changes will be saved when you update.</small>
                            </div>
                            <div class="d-flex flex-wrap gap-3" id="existingImagesContainer">
                                <!-- Existing images will be dynamically inserted here -->
                            </div>
                        </div>
                    </div>

                    <!-- Add New Images Section -->
                    <div class="row">
                        <div class="mb-3 col-md-6">
                            <label for="editSubImages" class="form-label fw-semibold">Add More Images</label>
                            <input type="file" class="form-control" id="editSubImages" name="newImages[]" accept="image/*" multiple>
                            <small class="text-muted">Accepted: JPG, PNG, GIF — Max: 5MB per image</small>
                            <div class="mt-2" id="newImagesPreview"></div>
                        </div>
                    </div>
                </div>

                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Sub-Workout</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- ========================== EDIT WORKOUT MODAL ========================== -->
<div class="modal fade" id="editWorkoutModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content" style="border-radius:16px; border:1px solid #dee2e6; box-shadow:0 8px 24px rgba(0,0,0,0.15); background:#fff;">
            <form id="editWorkoutForm" enctype="multipart/form-data">
                <div class="modal-header bg-light">
                    <h5 class="modal-title">Edit Workout Plan</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>

                <div class="modal-body" style="padding:2rem;">
                    <input type="hidden" id="editPlanId" name="planId">

                    <div class="mb-3">
                        <label class="form-label">Title</label>
                        <input type="text" class="form-control" id="editPlanTitle" name="planTitle" maxlength="60" required>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea class="form-control" id="editPlanDescription" name="planDescription" rows="3"></textarea>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Billing Options</label>
                        <div class="form-control bg-light">Monthly and Yearly</div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3" id="monthlyPriceDiv">
                            <label class="form-label">Monthly Price</label>
                            <input type="number" class="form-control" id="editMonthlyPrice" name="monthlyPrice" step="0.01" min="0" max="9999.99" required>
                        </div>
                        <div class="col-md-6 mb-3" id="annualPriceDiv">
                            <label class="form-label">Annual Price</label>
                            <input type="number" class="form-control" id="editAnnualPrice" name="annualPrice" step="0.01" min="0" max="99999.99" required>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Workout Type</label>
                        <select class="form-select" id="editWorkoutType" name="workout_type_id">
                            <?php foreach ($workoutTypes as $type): ?>
                                <option value="<?= $type['id'] ?>"><?= htmlspecialchars($type['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Thumbnail</label>
                        <div><img id="currentThumbnail" src="" style="max-width:100px; border-radius:6px; margin-bottom:8px; border:1px solid #ddd;"></div>
                        <input type="file" class="form-control" name="planThumbnail" id="editPlanThumbnail" accept="image/*">
                    </div>
                </div>

                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- ========================== INVITE MODAL ========================== -->
<div class="modal fade" id="inviteModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <form id="inviteForm">
            <div class="modal-content">
                <div class="modal-header bg-light">
                    <h5 class="modal-title">Invite User to Workout</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="plan_id" value="<?= htmlspecialchars($plan['id']) ?>">
                    <div class="mb-3">
                        <label class="form-label">User Email</label>
                        <input type="email" name="user_email" id="user_email" class="form-control" required>
                    </div>
                    <div id="inviteMessage" class="mt-2 small"></div>
                </div>
                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Send Invite</button>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- ========================== FULL SCREEN IMAGE MODAL ========================== -->
<div class="modal fade" id="fullScreenImageModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-fullscreen">
        <div class="modal-content bg-dark">
            <div class="modal-header border-0">
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body d-flex justify-content-center align-items-center p-0">
                <div id="fullScreenCarousel" class="carousel slide w-100 h-100" data-bs-ride="false">
                    <div class="carousel-inner h-100" id="fullScreenCarouselInner"></div>
                    <button class="carousel-control-prev" type="button" data-bs-target="#fullScreenCarousel" data-bs-slide="prev">
                        <span class="carousel-control-prev-icon" aria-hidden="true"></span>
                        <span class="visually-hidden">Previous</span>
                    </button>
                    <button class="carousel-control-next" type="button" data-bs-target="#fullScreenCarousel" data-bs-slide="next">
                        <span class="carousel-control-next-icon" aria-hidden="true"></span>
                        <span class="visually-hidden">Next</span>
                    </button>
                </div>
            </div>
            <div class="modal-footer border-0 justify-content-center">
                <span id="imageCounter" class="text-white fs-5"></span>
            </div>
        </div>
    </div>
</div>

<!-- ========================== WORKOUT DETAILS ========================== -->
<div class="row">
    <div class="col-md-6">
        <img src="<?= !empty($plan['thumbnail']) ? htmlspecialchars($plan['thumbnail']) : base_url('dashboard-assets/images/img/Mask.png') ?>"
            class="img-fluid rounded" style="max-height: 280px; object-fit: cover;"
            alt="<?= htmlspecialchars($plan['title']) ?>">
    </div>
    <div class="col-md-6">
        <h4><?= htmlspecialchars($plan['title']) ?></h4>
        <p><strong>Type:</strong> <?= htmlspecialchars($plan['type']) ?></p>
        <p><strong>Description:</strong><br><?= htmlspecialchars($plan['description']) ?></p>
        <p><strong>Sub Workouts:</strong> <?= count($plan['subWorkouts']) ?></p>
    </div>
</div>

<hr>

<!-- ========================== SUB WORKOUTS DISPLAY ========================== -->
<h5 class="mt-4">Sub Workouts</h5>
<div class="row mt-3" id="subWorkoutList">
    <?php foreach ($plan['subWorkouts'] as $sub): ?>
        <div class="col-md-4 mb-4 sub-workout-item" data-id="<?= $sub['id'] ?>">
            <div class="card sub-card h-100 shadow-sm border-0 position-relative">

                <!-- Thumbnail Section with Video + Images -->
                <div class="sub-video-wrapper position-relative">

                    <!-- VIDEO SECTION (if video exists) -->
                    <?php if (!empty($sub['video'])): ?>
                        <div class="video-section h-100">
                            <video src="<?= htmlspecialchars($sub['video']) ?>"
                                class="sub-video d-none w-100 h-100"
                                preload="metadata"
                                playsinline
                                controls></video>

                            <!-- Video Thumbnail Overlay -->
                            <div class="video-thumbnail-overlay h-100">
                                <img src="<?= !empty($sub['imageUrls'][0]) ? htmlspecialchars($sub['imageUrls'][0]) : base_url('dashboard-assets/images/img/Mask.png') ?>"
                                    alt="Video thumbnail"
                                    class="w-100 h-100"
                                    style="object-fit: cover;">
                                <div class="video-overlay d-flex justify-content-center align-items-center">
                                    <i class="bi bi-play-circle-fill play-icon"></i>
                                </div>
                            </div>
                        </div>

                        <!-- VIEW IMAGES BUTTON (if images exist with video) -->
                        <?php if (!empty($sub['imageUrls']) && count($sub['imageUrls']) > 0): ?>
                            <button class="btn btn-sm btn-light position-absolute bottom-0 start-50 translate-middle-x mb-2 view-images-btn"
                                onclick="openFullScreenImageModal('<?= $sub['id'] ?>', 0)"
                                style="z-index: 15; opacity: 0.95;">
                                <i class="bi bi-images me-1"></i>
                                View <?= count($sub['imageUrls']) ?> Image<?= count($sub['imageUrls']) > 1 ? 's' : '' ?>
                            </button>
                        <?php endif; ?>
                    <?php endif; ?>

                    <!-- IMAGES CAROUSEL SECTION (Only if NO video) -->
                    <?php if (empty($sub['video'])): ?>
                        <div class="images-carousel-section position-relative">
                            <?php if (!empty($sub['imageUrls']) && count($sub['imageUrls']) > 0): ?>
                                <div id="carousel-<?= $sub['id'] ?>" class="carousel slide h-100" data-bs-ride="false">
                                    <?php if (count($sub['imageUrls']) > 1): ?>
                                        <div class="carousel-indicators">
                                            <?php foreach ($sub['imageUrls'] as $index => $imageUrl): ?>
                                                <button type="button" data-bs-target="#carousel-<?= $sub['id'] ?>"
                                                    data-bs-slide-to="<?= $index ?>"
                                                    class="<?= $index === 0 ? 'active' : '' ?>"
                                                    aria-label="Slide <?= $index + 1 ?>"></button>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endif; ?>

                                    <div class="carousel-inner h-100">
                                        <?php foreach ($sub['imageUrls'] as $index => $imageUrl): ?>
                                            <div class="carousel-item h-100 <?= $index === 0 ? 'active' : '' ?>">
                                                <img src="<?= htmlspecialchars($imageUrl) ?>"
                                                    class="d-block w-100 h-100 carousel-image clickable-image"
                                                    style="object-fit: cover; cursor: pointer;"
                                                    alt="Workout image <?= $index + 1 ?>"
                                                    onclick="openFullScreenImageModal('<?= $sub['id'] ?>', <?= $index ?>)">
                                            </div>
                                        <?php endforeach; ?>
                                    </div>

                                    <?php if (count($sub['imageUrls']) > 1): ?>
                                        <button class="carousel-control-prev" type="button" data-bs-target="#carousel-<?= $sub['id'] ?>" data-bs-slide="prev">
                                            <span class="carousel-control-prev-icon" aria-hidden="true"></span>
                                            <span class="visually-hidden">Previous</span>
                                        </button>
                                        <button class="carousel-control-next" type="button" data-bs-target="#carousel-<?= $sub['id'] ?>" data-bs-slide="next">
                                            <span class="carousel-control-next-icon" aria-hidden="true"></span>
                                            <span class="visually-hidden">Next</span>
                                        </button>
                                    <?php endif; ?>
                                </div>

                                <?php if (count($sub['imageUrls']) > 1): ?>
                                    <div class="image-count-badge">
                                        <span class="badge bg-dark bg-opacity-75">
                                            <i class="bi bi-images me-1"></i><?= count($sub['imageUrls']) ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                            <?php else: ?>
                                <div class="no-images-placeholder d-flex justify-content-center align-items-center h-100 bg-light">
                                    <div class="text-center text-muted">
                                        <i class="bi bi-image display-6"></i>
                                        <p class="mt-2 small">No images available</p>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                    <!-- Edit/Delete Buttons -->
                    <div class="position-absolute top-0 end-0 d-flex flex-column p-2 control-buttons" style="z-index: 20;">
                        <button class="btn btn-sm btn-light mb-1"
                            onclick="openEditSubWorkoutModal('<?= $plan['id'] ?>', '<?= $sub['id'] ?>')"
                            title="Edit Sub-Workout">
                            <i class="bi bi-pencil-square text-primary"></i>
                        </button>
                        <button class="btn btn-sm btn-light"
                            onclick="confirmDeleteSubWorkout('<?= $plan['id'] ?>', '<?= $sub['id'] ?>')"
                            title="Delete Sub-Workout">
                            <i class="bi bi-trash text-danger"></i>
                        </button>
                    </div>
                </div>

                <!-- Card Body -->
                <div class="card-body d-flex flex-column justify-content-between">
                    <div>
                        <h6 class="fw-bold mb-1 text-truncate"><?= htmlspecialchars($sub['title']) ?></h6>
                        <p class="small text-muted mb-2 text-truncate-2"><?= htmlspecialchars($sub['description'] ?? 'No description provided') ?></p>
                    </div>
                    <div class="d-flex justify-content-between align-items-center mt-2">
                        <span class="badge bg-success-subtle text-success fw-medium px-2 py-1">
                            <i class="bi bi-clock me-1"></i>
                            <?= $formatDuration($sub['duration'] ?? 0) ?>
                        </span>
                        <span class="badge bg-light text-dark fw-medium px-2 py-1">
                            <i class="bi bi-list-ol me-1"></i>#<?= htmlspecialchars($sub['order'] ?? 0) ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>
    <?php endforeach; ?>
</div>

<!-- ========================== JS SCRIPTS ========================== -->
<script src="https://cdn.jsdelivr.net/npm/jquery@3.7.1/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
    const base_url = "<?= base_url() ?>";
    const formatDuration = (seconds) => {
        const safeSeconds = Math.max(0, parseInt(seconds, 10) || 0);
        const mins = Math.floor(safeSeconds / 60);
        const secs = safeSeconds % 60;
        return `${mins}:${secs.toString().padStart(2, "0")}`;
    };

    /* ========================================================================== */
    /*                         ADD SUB WORKOUT MODAL                              */
    /* ========================================================================== */
    let selectedImages = [];

    function openAddSubWorkoutModal(planId) {
        $("#parentPlanId").val(planId);
        $("#addSubWorkoutForm")[0].reset();
        $("#videoPreview, #selectedImagesContainer").html("");
        $("#videoError, #subImagesError").text("");
        $("#subWorkoutDuration").val(0);
        videoDuration = 0;
        selectedImages = [];
        $("#addSubWorkoutModal").modal("show");
    }

    // Video preview + duration extraction
    let videoDuration = 0;
    $("#video").on("change", function(e) {
        const file = e.target.files[0];
        if (!file) {
            videoDuration = 0;
            $("#subWorkoutDuration").val(0);
            return;
        }

        const videoURL = URL.createObjectURL(file);
        const videoElement = document.createElement("video");
        videoElement.preload = "metadata";
        videoElement.src = videoURL;

        videoElement.onloadedmetadata = function() {
            videoDuration = Math.round(videoElement.duration || 0);
            $("#subWorkoutDuration").val(videoDuration);

            $("#videoPreview").html(`
            <video width="100%" height="auto" controls style="border-radius:8px; border:1px solid #ccc;">
                <source src="${videoURL}" type="${file.type}">
            </video>
            <small class="text-muted d-block mt-1">Duration: ${formatDuration(videoDuration)}</small>
        `);
        };
    });

    // Multiple images preview for ADD modal
    $("#subImages").on("change", function(e) {
        const files = Array.from(e.target.files);

        files.forEach(file => {
            const error = validateFile(file, ['jpeg', 'jpg', 'png', 'gif'], 5, "Image");
            if (error) {
                $("#subImagesError").text(error);
                return;
            }

            if (!selectedImages.some(img => img.name === file.name && img.size === file.size)) {
                selectedImages.push(file);
                addImagePreview(file, selectedImages.length - 1);
            }
        });

        this.value = "";
    });

    function addImagePreview(file, index) {
        const previewContainer = $("#selectedImagesContainer");
        const previewItem = $(`
        <div class="image-preview-item" data-index="${index}">
            <img src="${URL.createObjectURL(file)}" style="width:100px; height:100px; object-fit:cover; border-radius:6px;">
            <button type="button" class="remove-btn" style="position:absolute; top:2px; right:2px; background:rgba(255,0,0,0.7); color:white; border:none; border-radius:50%; width:20px; height:20px; font-size:12px; cursor:pointer;">×</button>
        </div>
    `);

        previewItem.find('.remove-btn').click(() => {
            selectedImages.splice(index, 1);
            refreshImagePreviews();
        });

        previewContainer.append(previewItem);
    }

    function refreshImagePreviews() {
        $("#selectedImagesContainer").empty();
        selectedImages.forEach((file, index) => addImagePreview(file, index));
    }

    // ADD Sub-Workout Form Submission
    $("#addSubWorkoutForm").on("submit", async function(e) {
        e.preventDefault();

        const formData = new FormData(this);

        // Append multiple images
        selectedImages.forEach((image, index) => {
            formData.append(`images[${index}]`, image);
        });

        Swal.fire({
            title: 'Adding Sub-Workout...',
            text: 'Please wait while we upload your files.',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await fetch(`${base_url}workout/addSubWorkout`, {
                method: "POST",
                body: formData
            });
            const data = await res.json();
            Swal.close();

            if (data.status === "success") {
                Swal.fire({
                    icon: "success",
                    title: "Added!",
                    text: data.message || "Sub-workout added successfully.",
                    showConfirmButton: false,
                    timer: 1500
                }).then(() => location.reload());
            } else {
                Swal.fire("Failed", data.message || "Could not add sub-workout.", "error");
            }
        } catch (err) {
            Swal.close();
            Swal.fire("Error", "Something went wrong.", "error");
            console.error(err);
        }
    });

    /* ========================================================================== */
    /*                         EDIT SUB WORKOUT MODAL                             */
    /* ========================================================================== */
    let existingImages = [];
    let imagesToDelete = [];
    let newImages = [];

    function openEditSubWorkoutModal(planId, subId) {
        existingImages = [];
        imagesToDelete = [];
        newImages = [];

        // Get sub-workout data from PHP
        const subWorkoutData = <?= json_encode($plan['subWorkouts']) ?>;
        const subWorkout = subWorkoutData.find(sub => sub.id === subId);

        if (!subWorkout) {
            Swal.fire('Error', 'Sub-workout not found', 'error');
            return;
        }

        $("#editPlanIdSub").val(planId);
        $("#editSubId").val(subId);
        $("#editSubTitle").val(subWorkout.title);
        $("#editSubDescription").val(subWorkout.description || '');
        const existingDurationSeconds = Math.max(0, parseInt(subWorkout.duration, 10) || 0);
        $("#editSubDuration").val(existingDurationSeconds);

        // Handle existing images
        const existingImagesContainer = $("#existingImagesContainer");
        existingImagesContainer.empty();

        // Get all images from imageUrls array
        if (subWorkout.imageUrls && Array.isArray(subWorkout.imageUrls) && subWorkout.imageUrls.length > 0) {
            subWorkout.imageUrls.forEach((imgSrc, index) => {
                if (imgSrc && !imgSrc.includes('Mask.png')) {
                    existingImages.push(imgSrc);
                    addExistingImageItem(imgSrc, index);
                }
            });
        }

        // If no images found, show message
        if (existingImages.length === 0) {
            existingImagesContainer.html('<div class="text-muted text-center py-3"><i class="bi bi-image me-2"></i>No images available</div>');
        }

        $("#existingImagesData").val(JSON.stringify(existingImages));

        // Video preview
        if (subWorkout.video) {
            $("#editVideoPreview").html(`
            <video width="100%" height="auto" controls style="border-radius:8px; border:1px solid #ccc;">
                <source src="${subWorkout.video}" type="video/mp4">
            </video>
            <small class="text-muted d-block mt-1">Current duration: ${formatDuration(existingDurationSeconds)}</small>
            <small class="text-muted d-block">Replace the video to update duration automatically.</small>
        `);
        } else {
            $("#editVideoPreview").html('<p class="text-muted">No video uploaded</p>');
        }

        $("#newImagesPreview").empty();
        $("#editSubWorkoutModal").modal("show");
    }

    function addExistingImageItem(imgSrc, index) {
        const existingImagesContainer = $("#existingImagesContainer");
        const imageItem = $(`
        <div class="existing-image-item position-relative" data-index="${index}" data-url="${imgSrc}" style="display: inline-block;">
            <img src="${imgSrc}" 
                 style="width:120px; height:120px; object-fit:cover; border-radius:8px; border:2px solid #dee2e6;">
            <button type="button" class="btn btn-sm btn-danger position-absolute top-0 end-0 m-1 remove-existing-image" 
                    style="width:24px; height:24px; border-radius:50%; padding:0; display:flex; align-items:center; justify-content:center;">
                <i class="bi bi-trash" style="font-size:12px;"></i>
            </button>
            <div class="text-center small mt-1">Image ${index + 1}</div>
        </div>
    `);
        existingImagesContainer.append(imageItem);
    }

    // Remove existing image
    $(document).on('click', '.remove-existing-image', function() {
        const imageItem = $(this).closest('.existing-image-item');
        const imageUrl = imageItem.data('url');

        Swal.fire({
            title: 'Remove Image?',
            text: 'This image will be removed from the sub-workout.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, remove it',
            cancelButtonText: 'Cancel',
            confirmButtonColor: '#d33'
        }).then((result) => {
            if (result.isConfirmed) {
                // Add to deletion list
                imagesToDelete.push(imageUrl);

                // Remove from existing images array
                const urlIndex = existingImages.indexOf(imageUrl);
                if (urlIndex > -1) {
                    existingImages.splice(urlIndex, 1);
                }

                imageItem.fadeOut(300, function() {
                    $(this).remove();
                    updateExistingImagesDisplay();
                });

                Swal.fire('Removed!', 'Image will be removed when you save changes.', 'success');
            }
        });
    });

    function updateExistingImagesDisplay() {
        const container = $("#existingImagesContainer");
        if (container.children().length === 0) {
            container.html('<div class="text-muted text-center py-3"><i class="bi bi-image me-2"></i>No images remaining</div>');
        }
    }
    // New images preview for EDIT modal
    $("#editSubImages").on("change", function(e) {
        const files = Array.from(e.target.files);

        files.forEach(file => {
            const error = validateFile(file, ['jpeg', 'jpg', 'png', 'gif'], 5, "Image");
            if (error) {
                Swal.fire('Error', error, 'error');
                return;
            }

            if (!newImages.some(img => img.name === file.name && img.size === file.size)) {
                newImages.push(file);
                addNewImagePreview(file, newImages.length - 1);
            }
        });

        this.value = "";
    });

    function addNewImagePreview(file, index) {
        const previewContainer = $("#newImagesPreview");
        const previewItem = $(`
        <div class="new-image-preview-item position-relative d-inline-block me-2 mb-2" data-index="${index}">
            <img src="${URL.createObjectURL(file)}" 
                 style="width:100px; height:100px; object-fit:cover; border-radius:6px; border:2px solid #28a745;">
            <button type="button" class="btn btn-sm btn-danger position-absolute top-0 end-0 m-1 remove-new-image" 
                    style="width:20px; height:20px; border-radius:50%; padding:0; display:flex; align-items:center; justify-content:center;">
                <i class="bi bi-x" style="font-size:10px;"></i>
            </button>
            <div class="text-center small mt-1" style="font-size:11px;">New ${index + 1}</div>
        </div>
    `);
        previewContainer.append(previewItem);
    }

    $(document).on('click', '.remove-new-image', function() {
        const previewItem = $(this).closest('.new-image-preview-item');
        const index = parseInt(previewItem.data('index'));

        newImages.splice(index, 1);
        previewItem.fadeOut(300, function() {
            $(this).remove();
            updateNewImagesPreview();
        });
    });

    function updateNewImagesPreview() {
        const container = $("#newImagesPreview");
        container.empty();
        newImages.forEach((file, index) => addNewImagePreview(file, index));
    }

    // Video preview for EDIT modal
    $("#editSubVideo").on("change", function(e) {
        const file = e.target.files[0];
        if (!file) {
            return;
        }

        const videoURL = URL.createObjectURL(file);
        const videoElement = document.createElement("video");
        videoElement.preload = "metadata";
        videoElement.src = videoURL;

        videoElement.onloadedmetadata = function() {
            const durationInSeconds = Math.round(videoElement.duration || 0);
            $("#editSubDuration").val(durationInSeconds);

            $("#editVideoPreview").html(`
            <video width="100%" height="auto" controls style="border-radius:8px; border:1px solid #ccc;">
                <source src="${videoURL}" type="${file.type}">
            </video>
            <small class="text-muted d-block mt-1">Duration: ${formatDuration(durationInSeconds)}</small>
        `);
        };
    });

    // EDIT Sub-Workout Form Submission
    $("#editSubWorkoutForm").on("submit", async function(e) {
        e.preventDefault();

        const remainingExistingImages = $("#existingImagesContainer").children('.existing-image-item').length;
        if (remainingExistingImages === 0 && newImages.length === 0) {
            Swal.fire('Error', 'At least one image is required for the sub-workout.', 'error');
            return;
        }

        const formData = new FormData(this);

        // Add images to delete
        imagesToDelete.forEach((imageUrl, index) => {
            formData.append(`imagesToDelete[${index}]`, imageUrl);
        });

        // Add new images
        newImages.forEach((image, index) => {
            formData.append(`newImages[${index}]`, image);
        });

        Swal.fire({
            title: "Updating Sub-Workout...",
            text: "Please wait while we upload changes.",
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await fetch(`${base_url}workout/updateSubWorkout`, {
                method: "POST",
                body: formData
            });
            const data = await res.json();
            Swal.close();

            if (data.status === "success") {
                Swal.fire({
                    icon: "success",
                    title: "Updated!",
                    text: data.message || "Sub-workout updated successfully.",
                    showConfirmButton: false,
                    timer: 1500
                }).then(() => location.reload());
            } else {
                Swal.fire("Error", data.message || "Failed to update sub-workout.", "error");
            }
        } catch (err) {
            Swal.close();
            Swal.fire("Error", "Something went wrong while updating.", "error");
            console.error(err);
        }
    });

    /* ========================================================================== */
    /*                            EDIT WORKOUT MODAL                              */
    /* ========================================================================== */
    async function openEditModal(planId) {
        try {
            const res = await fetch(`${base_url}workout/getPlan/${planId}`);
            const data = await res.json();

            if (data.status === "success") {
                const plan = data.plan;
                const types = data.workoutTypes;

                $("#editPlanId").val(plan.id);
                $("#editPlanTitle").val(plan.title);
                $("#editPlanDescription").val(plan.description);
                $("#editMonthlyPrice").val(plan.monthlyPrice !== null && plan.monthlyPrice !== undefined ? plan.monthlyPrice : "");
                $("#editAnnualPrice").val(plan.annualPrice !== null && plan.annualPrice !== undefined ? plan.annualPrice : "");
                $("#currentThumbnail").attr("src", plan.thumbnail || "");

                const select = $("#editWorkoutType").empty();
                types.forEach(t => select.append(`<option value="${t.id}" ${t.id === plan.workout_type_id ? 'selected' : ''}>${t.name}</option>`));

                $("#editWorkoutModal").modal("show");
            } else {
                Swal.fire("Not Found", "Workout plan could not be loaded.", "error");
            }
        } catch (err) {
            console.error(err);
            Swal.fire("Error", "Could not load workout details.", "error");
        }
    }

    $("#editPlanThumbnail").on("change", e => {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = e => $("#currentThumbnail").attr("src", e.target.result);
            reader.readAsDataURL(file);
        }
    });

    $("#editWorkoutForm").on("submit", async e => {
        e.preventDefault();
        const formData = new FormData(e.target);
        const planId = formData.get("planId");
        const planTitle = String(formData.get("planTitle") || "").trim();
        const monthlyPrice = Number(formData.get("monthlyPrice"));
        const annualPrice = Number(formData.get("annualPrice"));

        if (!planTitle) {
            Swal.fire("Validation Error", "Title is required.", "error");
            return;
        }
        if (planTitle.length > 60) {
            Swal.fire("Validation Error", "Title must be 60 characters or less.", "error");
            return;
        }
        formData.set("planTitle", planTitle);

        if (!Number.isFinite(monthlyPrice) || monthlyPrice < 0) {
            Swal.fire("Validation Error", "Please enter a valid monthly price.", "error");
            return;
        }
        if (!Number.isFinite(annualPrice) || annualPrice < 0) {
            Swal.fire("Validation Error", "Please enter a valid annual price.", "error");
            return;
        }
        if (annualPrice <= monthlyPrice) {
            Swal.fire("Validation Error", "Annual price should be greater than monthly price.", "error");
            return;
        }
        try {
            const res = await fetch(`${base_url}workout/update/${planId}`, {
                method: "POST",
                body: formData
            });
            const result = await res.json();

            if (result.status === "success") {
                Swal.fire({
                    icon: "success",
                    title: "Workout Updated!",
                    showConfirmButton: false,
                    timer: 1500
                }).then(() => location.reload());
            } else {
                Swal.fire("Update Failed", result.message || "Something went wrong.", "error");
            }
        } catch (err) {
            Swal.fire("Error", "Unable to update workout plan.", "error");
        }
    });

    /* ========================================================================== */
    /*                            DELETE FUNCTIONS                                */
    /* ========================================================================== */
    async function confirmDeletePlan(planId) {
        const confirm = await Swal.fire({
            title: "Are you sure?",
            text: "This will permanently delete the workout plan.",
            icon: "warning",
            showCancelButton: true,
            confirmButtonText: "Yes, delete it",
            cancelButtonText: "Cancel",
            confirmButtonColor: "#d33"
        });

        if (confirm.isConfirmed) {
            Swal.showLoading();
            try {
                const res = await fetch(`${base_url}workout/delete/${planId}`, {
                    method: "POST"
                });
                const data = await res.json();

                if (data.status === "success") {
                    Swal.fire({
                        icon: "success",
                        title: "Deleted!",
                        showConfirmButton: false,
                        timer: 1500
                    }).then(() => window.location.href = `${base_url}dashboard/workout-plans`);
                } else {
                    Swal.fire("Delete Failed", data.message || "Unable to delete.", "error");
                }
            } catch {
                Swal.fire("Error", "Something went wrong while deleting.", "error");
            }
        }
    }

    async function confirmDeleteSubWorkout(planId, subId) {
        const confirm = await Swal.fire({
            title: "Delete Sub-Workout?",
            text: "This will permanently remove this sub-workout from the plan.",
            icon: "warning",
            showCancelButton: true,
            confirmButtonText: "Yes, delete it",
            cancelButtonText: "Cancel",
            confirmButtonColor: "#d33"
        });

        if (!confirm.isConfirmed) return;

        Swal.fire({
            title: "Deleting...",
            text: "Please wait",
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await fetch(`${base_url}workout/deleteSubWorkout/${planId}/${subId}`, {
                method: "POST"
            });
            const data = await res.json();
            Swal.close();

            if (data.status === "success") {
                Swal.fire({
                    icon: "success",
                    title: "Sub-Workout Deleted!",
                    showConfirmButton: false,
                    timer: 1200
                }).then(() => {
                    $(`.sub-workout-item[data-id='${subId}']`).fadeOut(300, function() {
                        $(this).remove();
                    });
                });
            } else {
                Swal.fire("Error", data.message || "Failed to delete sub-workout.", "error");
            }
        } catch (err) {
            Swal.close();
            Swal.fire("Error", "Something went wrong during deletion.", "error");
        }
    }

    /* ========================================================================== */
    /*                          VIDEO PLAYBACK HANDLING                           */
    /* ========================================================================== */
    document.addEventListener("click", function(e) {
        const card = e.target.closest(".sub-video-wrapper");
        if (!card) return;

        const videoThumbnail = card.querySelector(".video-thumbnail-overlay");
        const video = card.querySelector(".sub-video");

        if (videoThumbnail && video && e.target.closest(".video-overlay")) {
            videoThumbnail.classList.add("d-none");
            video.classList.remove("d-none");
            video.play();
        }
    });

    /* ========================================================================== */
    /*                       FULL SCREEN IMAGE VIEWER                             */
    /* ========================================================================== */
    function openFullScreenImageModal(subWorkoutId, startIndex = 0) {
        const subCard = $(`.sub-workout-item[data-id='${subWorkoutId}']`);
        const fullScreenCarouselInner = $('#fullScreenCarouselInner');
        fullScreenCarouselInner.empty();

        let imageUrls = [];

        // Check if there's a carousel (no video case)
        const carousel = subCard.find('.carousel-inner');
        if (carousel.length > 0) {
            const carouselItems = carousel.find('.carousel-item');
            carouselItems.each(function() {
                const imgSrc = $(this).find('img').attr('src');
                if (imgSrc) {
                    imageUrls.push(imgSrc);
                }
            });
        } else {
            // Video case - get images from PHP data
            const subWorkoutData = <?= json_encode($plan['subWorkouts']) ?>;
            const subWorkout = subWorkoutData.find(sub => sub.id === subWorkoutId);

            if (subWorkout && subWorkout.imageUrls) {
                imageUrls = subWorkout.imageUrls;
            }
        }

        // If no images found, show error
        if (imageUrls.length === 0) {
            Swal.fire('No Images', 'No images available to display.', 'info');
            return;
        }

        // Create carousel items
        imageUrls.forEach((imgSrc, index) => {
            const isActive = index === startIndex ? 'active' : '';
            const carouselItem = $(`
            <div class="carousel-item h-100 ${isActive}">
                <div class="d-flex justify-content-center align-items-center h-100">
                    <img src="${imgSrc}" 
                         class="img-fluid max-vh-100 max-vw-100"
                         style="object-fit: contain; max-height: 90vh; max-width: 90vw;"
                         alt="Workout image ${index + 1}">
                </div>
            </div>
        `);
            fullScreenCarouselInner.append(carouselItem);
        });

        updateImageCounter(startIndex + 1, imageUrls.length);

        const fullScreenModal = new bootstrap.Modal(document.getElementById('fullScreenImageModal'));
        fullScreenModal.show();

        $('#fullScreenCarousel').off('slid.bs.carousel').on('slid.bs.carousel', function(e) {
            const activeIndex = e.to;
            updateImageCounter(activeIndex + 1, imageUrls.length);
        });
    }

    function updateImageCounter(current, total) {
        $('#imageCounter').text(`${current} / ${total}`);
    }

    document.addEventListener('keydown', function(e) {
        const fullScreenModal = document.getElementById('fullScreenImageModal');
        if (fullScreenModal.classList.contains('show')) {
            const carousel = bootstrap.Carousel.getInstance(document.getElementById('fullScreenCarousel'));

            switch (e.key) {
                case 'ArrowLeft':
                    if (carousel) carousel.prev();
                    e.preventDefault();
                    break;
                case 'ArrowRight':
                    if (carousel) carousel.next();
                    e.preventDefault();
                    break;
                case 'Escape':
                    bootstrap.Modal.getInstance(fullScreenModal).hide();
                    e.preventDefault();
                    break;
            }
        }
    });
    /* ========================================================================== */
    /*                           INVITE FORM                                      */
    /* ========================================================================== */
    $("#inviteForm").on("submit", e => {
        e.preventDefault();
        const $btn = $(e.target).find("button[type='submit']");
        $btn.prop("disabled", true).text("Sending...");

        $.ajax({
            url: `${base_url}invite/send`,
            method: "POST",
            data: $(e.target).serialize(),
            dataType: "json",
            success: res => {
                const msg = $("#inviteMessage");
                if (res.status === "success") {
                    msg.removeClass("text-danger").addClass("text-success").text(res.message);
                    $("#user_email").val("");
                } else {
                    msg.removeClass("text-success").addClass("text-danger").text(res.message);
                }
                setTimeout(() => msg.text(""), 3000);
            },
            complete: () => $btn.prop("disabled", false).text("Send Invite")
        });
    });

    /* ========================================================================== */
    /*                           HELPER FUNCTIONS                                 */
    /* ========================================================================== */
    function validateFile(file, allowedTypes, maxSize, fieldName) {
        const fileType = file.type.toLowerCase();
        const fileSize = file.size;
        const maxSizeBytes = maxSize * 1024 * 1024;

        if (!allowedTypes.some(type => fileType.includes(type))) {
            return `Invalid file type for ${fieldName}. Allowed types: ${allowedTypes.join(', ')}`;
        }

        if (fileSize > maxSizeBytes) {
            return `${fieldName} must be less than ${maxSize}MB`;
        }

        return null;
    }
</script>

<style>
    @media (max-width: 767px) {
        .btn {
            padding: 6px 10px;
            font-size: 13px;
        }
    }

    .sub-card {
        border-radius: 14px;
        overflow: hidden;
        transition: all 0.25s ease;
        background: #fff;
    }

    .sub-card:hover {
        transform: translateY(-6px);
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
    }

    .sub-video-wrapper {
        position: relative;
        overflow: hidden;
        height: 190px;
        border-bottom: 1px solid #f0f0f0;
    }

    .sub-video,
    .carousel-image {
        width: 100%;
        height: 100%;
        object-fit: cover;
        transition: transform 0.3s ease;
    }

    .sub-card:hover .sub-video,
    .sub-card:hover .carousel-image {
        transform: scale(1.05);
    }

    .video-thumbnail-overlay {
        position: relative;
        width: 100%;
        height: 100%;
    }

    .video-overlay {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: linear-gradient(to top, rgba(0, 0, 0, 0.55) 10%, rgba(0, 0, 0, 0.1) 60%, transparent 100%);
        opacity: 0;
        transition: opacity 0.3s ease;
        cursor: pointer;
    }

    .video-thumbnail-overlay:hover .video-overlay {
        opacity: 1;
    }

    .play-icon {
        font-size: 2.8rem;
        color: rgba(255, 255, 255, 0.9);
        opacity: 0.85;
        transition: all 0.25s ease;
    }

    .play-icon:hover {
        transform: scale(1.1);
        opacity: 1;
    }

    .view-images-btn {
        border-radius: 20px;
        font-size: 0.85rem;
        font-weight: 500;
        padding: 6px 14px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
        transition: all 0.3s ease;
        border: 1px solid rgba(255, 255, 255, 0.3);
    }

    .view-images-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
        background: white !important;
    }

    .carousel {
        height: 100%;
        position: relative;
    }

    .carousel-inner,
    .carousel-item {
        height: 100%;
        position: relative;
    }

    .carousel-control-prev,
    .carousel-control-next {
        position: absolute;
        top: 50%;
        transform: translateY(-50%);
        width: 40px;
        height: 40px;
        background: rgba(0, 0, 0, 0.5);
        border-radius: 50%;
        opacity: 0;
        transition: opacity 0.3s ease;
        z-index: 5;
    }

    .carousel:hover .carousel-control-prev,
    .carousel:hover .carousel-control-next {
        opacity: 1;
    }

    .carousel-control-prev {
        left: 10px;
    }

    .carousel-control-next {
        right: 10px;
    }

    .carousel-control-prev-icon,
    .carousel-control-next-icon {
        width: 20px;
        height: 20px;
        background-size: 20px 20px;
    }

    .carousel-indicators {
        position: absolute;
        bottom: 10px;
        left: 50%;
        transform: translateX(-50%);
        margin: 0;
        z-index: 5;
    }

    .carousel-indicators button {
        width: 8px;
        height: 8px;
        border-radius: 50%;
        margin: 0 3px;
        border: none;
    }

    .control-buttons {
        z-index: 20;
    }

    .control-buttons button {
        border-radius: 8px;
        padding: 4px 6px;
        background: rgba(255, 255, 255, 0.95);
        border: none;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
        transition: all 0.2s ease;
    }

    .control-buttons button:hover {
        background: white;
        transform: translateY(-1px);
        box-shadow: 0 4px 8px rgba(0, 0, 0, 0.3);
    }

    .image-count-badge {
        position: absolute;
        top: 8px;
        left: 8px;
        z-index: 5;
    }

    .no-images-placeholder {
        height: 100%;
        background: #f8f9fa;
        border: 2px dashed #dee2e6;
    }

    .text-truncate-2 {
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }

    .modal-fullscreen .modal-content {
        background: rgba(0, 0, 0, 0.95) !important;
    }

    #fullScreenCarousel .carousel-control-prev,
    #fullScreenCarousel .carousel-control-next {
        width: 60px;
        height: 60px;
        background: rgba(255, 255, 255, 0.2);
        border-radius: 50%;
        opacity: 0.7;
        transition: all 0.3s ease;
    }

    #fullScreenCarousel .carousel-control-prev:hover,
    #fullScreenCarousel .carousel-control-next:hover {
        background: rgba(255, 255, 255, 0.3);
        opacity: 1;
    }

    #fullScreenCarousel .carousel-control-prev {
        left: 20px;
    }

    #fullScreenCarousel .carousel-control-next {
        right: 20px;
    }

    #fullScreenCarousel .carousel-control-prev-icon,
    #fullScreenCarousel .carousel-control-next-icon {
        width: 30px;
        height: 30px;
        background-size: 30px 30px;
    }

    #imageCounter {
        font-weight: 500;
        text-shadow: 0 2px 4px rgba(0, 0, 0, 0.5);
    }

    .clickable-image {
        cursor: zoom-in !important;
        transition: transform 0.2s ease;
    }

    .clickable-image:hover {
        transform: scale(1.02);
    }

    .modal-fullscreen .btn-close {
        filter: invert(1);
        opacity: 0.8;
        transition: opacity 0.3s ease;
    }

    .modal-fullscreen .btn-close:hover {
        opacity: 1;
    }
</style>

<?php $content = ob_get_clean();
$this->load->view('layouts/app', ['content' => $content]); ?>
