// === CONFIG ===
const DEFAULT_PROFILE_IMAGE =
  "<?= base_url('/dashboard-assets/images/avatars/11.png') ?>";
const coachId = "<?= esc($coachId) ?>";
const db = firebase.firestore();

// === DOM ELEMENTS ===
const chatHeader = document.getElementById("chatHeader");
const chatBox = document.getElementById("chatBox");
const messageInput = document.getElementById("messageInput");
const sendMessageBtn = document.getElementById("sendMessageBtn");
const chatList = document.getElementById("chatList");

// === ACTIVE CHAT STATE ===
let activeChatId = null;
let activeUserId = null;
let activeUserName = null;
let activeUserImage = null;
let chatCloseTimer = null;

// === FETCH USER INFO ===
async function getUserInfoById(userId) {
  try {
    const userDoc = await db.collection("Users").doc(userId).get();
    if (userDoc.exists) {
      const u = userDoc.data();
      return {
        name: u.userName || "User",
        image:
          u.image && u.image.trim() !== "" ? u.image : DEFAULT_PROFILE_IMAGE,
      };
    }
  } catch (e) {
    console.error("Error fetching user info:", e);
  }
  return { name: "User", image: DEFAULT_PROFILE_IMAGE };
}

// === OPEN CHAT ===
function openChat(chatId, otherUserId, otherUserName, otherUserImage) {
  if (
    !otherUserImage ||
    otherUserImage === "null" ||
    otherUserImage.trim() === ""
  ) {
    otherUserImage = DEFAULT_PROFILE_IMAGE;
  }

  loadMessages(chatId, otherUserId, otherUserName, otherUserImage);

  if (chatCloseTimer) clearTimeout(chatCloseTimer);
  chatCloseTimer = setTimeout(closeChat, 10000);

  messageInput.addEventListener(
    "input",
    () => {
      if (chatCloseTimer) clearTimeout(chatCloseTimer);
    },
    { once: true }
  );
}
window.openChat = openChat; // make global

// === CLOSE CHAT ===
function closeChat() {
  chatHeader.innerHTML = `<strong>Select a chat to start messaging</strong>`;
  chatBox.innerHTML = "";
  messageInput.value = "";
  messageInput.disabled = true;
  sendMessageBtn.disabled = true;
}

// === FETCH CHATS LIST ===
db.collection("Chats")
  .where(`participants.${coachId}`, "==", true)
  .onSnapshot(async (snapshot) => {
    if (snapshot.empty) {
      chatList.innerHTML = `<p class="p-3 text-muted">No chats found.</p>`;
      return;
    }

    let chats = snapshot.docs.map((doc) => ({ ...doc.data(), id: doc.id }));
    chats.sort(
      (a, b) =>
        (b.lastMessage?.timestamp || 0) - (a.lastMessage?.timestamp || 0)
    );

    chatList.innerHTML = "";

    for (let chat of chats) {
      const otherUserId = Object.keys(chat.participants).find(
        (id) => id !== coachId
      );
      const { name, image } = await getUserInfoById(otherUserId);
      const lastMessage = chat.lastMessage?.content || "No messages yet";

      const chatItem = document.createElement("div");
      chatItem.className =
        "chat-item p-2 d-flex align-items-center border-bottom";
      chatItem.style.cursor = "pointer";
      chatItem.innerHTML = `
          <img src="${image}" width="40" height="40" class="me-2 rounded-circle" 
               onerror="this.src='${DEFAULT_PROFILE_IMAGE}'">
          <div>
              <strong>${name}</strong><br>
              <small>${lastMessage}</small>
          </div>
        `;
      chatItem.onclick = () => openChat(chat.id, otherUserId, name, image);
      chatList.appendChild(chatItem);
    }
  });

// === LOAD MESSAGES ===
function loadMessages(chatId, otherUserId, otherUserName, otherUserImage) {
  activeChatId = chatId;
  activeUserId = otherUserId;
  activeUserName = otherUserName;
  activeUserImage = otherUserImage;

  chatHeader.innerHTML = `<strong>${activeUserName}</strong>`;
  messageInput.disabled = false;
  sendMessageBtn.disabled = false;

  db.collection("Chats")
    .doc(chatId)
    .collection("Conversations")
    .orderBy("timestamp", "asc")
    .onSnapshot((snapshot) => {
      chatBox.innerHTML = "";
      snapshot.forEach((doc) => {
        const msg = doc.data();
        const isSender = msg.fromID === coachId;
        const timeString = msg.timestamp
          ? new Date(msg.timestamp).toLocaleTimeString([], {
              hour: "2-digit",
              minute: "2-digit",
            })
          : "";

        chatBox.innerHTML += `
            <div class="message-row ${isSender ? "sent" : "received"}">
              <div class="message-bubble">
                ${msg.content}
                <div class="message-time">${timeString}</div>
              </div>
            </div>
          `;
      });
      chatBox.scrollTop = chatBox.scrollHeight;
    });
}

// === SEND MESSAGE ===
sendMessageBtn.onclick = async () => {
  const content = messageInput.value.trim();
  if (!content) return;
  if (chatCloseTimer) clearTimeout(chatCloseTimer);

  try {
    let chatIdToUse = activeChatId;
    const timestampMs = Date.now();

    if (!chatIdToUse) {
      const newChatRef = await db.collection("Chats").add({
        participants: { [coachId]: true, [activeUserId]: true },
        lastMessage: { content, timestamp: timestampMs },
        createdAt: timestampMs,
      });
      chatIdToUse = newChatRef.id;
      activeChatId = chatIdToUse;
    }

    const message = {
      content,
      fromID: coachId,
      toID: activeUserId,
      isRead: false,
      type: "text",
      timestamp: timestampMs,
    };

    await db
      .collection("Chats")
      .doc(chatIdToUse)
      .collection("Conversations")
      .add(message);
    await db
      .collection("Chats")
      .doc(chatIdToUse)
      .update({
        lastMessage: { content, timestamp: timestampMs },
      });

    messageInput.value = "";
  } catch (error) {
    console.error("Send error:", error);
  }
};
