<?php


defined('BASEPATH') or exit('No direct script access allowed');

class WorkoutController extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();

        // Load model
        $this->load->model('WorkoutModel');

        // Load session library
        $this->load->library('session');

        // Load Firebase library (make sure you have one for CI3)
        $this->load->library('firebase');

        // Cache driver (optional)
        $this->load->driver('cache');
    }

    public function index()
    {

        $workoutTypes = $this->WorkoutModel->getWorkoutTypes();
        // print_r($workoutTypes);
        // die;



        $this->load->view('pages/workout_plans', [
            'workoutTypes' => $workoutTypes
        ]);
    }


    public function addWorkoutPlan()
    {
        try {
            // 🔹 Capture and debug POST + FILES
            $postData = $this->input->post();


            // 🔹 Basic plan data
            $title           = trim($this->input->post('planTitle') ?? '');
            $description     = $this->input->post('planDescription') ?? '';
            $monthlyPrice    = $this->input->post('monthlyPrice') ?? '';
            $annualPrice     = $this->input->post('annualPrice') ?? '';
            $workout_type_id = $this->input->post('workout_type_id') ?? '';

            // 🔹 Parse subWorkouts properly
            $subWorkouts = $this->input->post('subWorkouts') ?? [];
            log_message('debug', 'SubWorkouts Raw: ' . print_r($subWorkouts, true));

            if (empty($subWorkouts) || !is_array($subWorkouts)) {
                return $this->jsonError("No workout blocks provided or invalid format");
            }

            // 🔹 Validation
            $errors = [];
            if (empty($title)) $errors['planTitle'] = 'Title is required';
            if (!empty($title) && strlen($title) > 60) $errors['planTitle'] = 'Title must be 60 characters or less';
            if (empty($description)) $errors['planDescription'] = 'Description is required';

            if (!is_numeric($monthlyPrice) || $monthlyPrice < 0) {
                $errors['monthlyPrice'] = 'Valid monthly price is required';
            }
            if (!is_numeric($annualPrice) || $annualPrice < 0) {
                $errors['annualPrice'] = 'Valid annual price is required';
            }
            if (!empty($errors)) {
                return $this->jsonError("Please fix the validation errors", $errors);
            }

            // 🔹 Firebase setup
            $firestore       = $this->firebase->getFirestore();
            $storageBucket   = $this->firebase->getStorage();
            $defaultImageUrl = base_url('dashboard-assets/images/img/Mask.png');
            $nowMillis       = round(microtime(true) * 1000);

            // 🔹 Create main workout plan
            $mainRef = $firestore->collection('workoutplan')->add([
                'title'           => $title,
                'description'     => $description,
                'monthlyPrice'    => floatval($monthlyPrice),
                'annualPrice'     => floatval($annualPrice),
                // Total duration is not stored on main plan; durations are kept per sub-workout
                'duration'        => 0,
                'workout_type_id' => $workout_type_id,
                'thumbnail'       => '',
                'userId'          => $this->session->userdata('user_doc_id') ?? 'demo_user',
                'createdAt'       => $nowMillis,
                'updatedAt'       => ''
            ]);

            $mainDocId    = $mainRef->id();
            $thumbnailSet = false;

            // 🔹 NEW: Handle multiple images upload structure
            log_message('debug', 'Checking for subWorkoutImages: ' . print_r($_FILES['subWorkoutImages'] ?? 'Not found', true));

            // 🔹 Loop through subWorkouts
            foreach ($subWorkouts as $index => $block) {
                $videoUrl = '';
                $imageUrls = []; // NEW: Store multiple image URLs

                log_message('debug', "Processing sub-workout $index: " . print_r($block, true));

                // --- Handle video upload ---
                if (isset($_FILES['videos']['tmp_name'][$index]) && !empty($_FILES['videos']['tmp_name'][$index])) {
                    $videoTmp = $_FILES['videos']['tmp_name'][$index];
                    $videoNameRaw = $_FILES['videos']['name'][$index];

                    if (is_uploaded_file($videoTmp)) {
                        $videoName = 'Videos/' . uniqid() . '_' . basename($videoNameRaw);
                        $storageBucket->upload(fopen($videoTmp, 'r'), ['name' => $videoName]);
                        $videoUrl = "https://firebasestorage.googleapis.com/v0/b/{$storageBucket->name()}/o/" . rawurlencode($videoName) . "?alt=media";
                        log_message('debug', "Video uploaded for block $index: $videoUrl");
                    }
                }

                // --- NEW: Handle multiple images upload for this sub-workout ---
                if (isset($_FILES['subWorkoutImages']['tmp_name'][$index]) && is_array($_FILES['subWorkoutImages']['tmp_name'][$index])) {
                    $imageFiles = $_FILES['subWorkoutImages'];

                    log_message('debug', "Multiple images found for sub-workout $index: " . count($imageFiles['tmp_name'][$index]));

                    // Process each image in the sub-array
                    foreach ($imageFiles['tmp_name'][$index] as $imgIndex => $tmpName) {
                        if (!empty($tmpName) && is_uploaded_file($tmpName)) {
                            $imageNameRaw = $imageFiles['name'][$index][$imgIndex];
                            $imageName = 'SubWorkoutImages/' . uniqid() . '_' . basename($imageNameRaw);

                            // Upload to Firebase Storage
                            $storageBucket->upload(fopen($tmpName, 'r'), ['name' => $imageName]);
                            $imageUrl = "https://firebasestorage.googleapis.com/v0/b/{$storageBucket->name()}/o/" . rawurlencode($imageName) . "?alt=media";

                            $imageUrls[] = $imageUrl;
                            log_message('debug', "Image $imgIndex uploaded for block $index: $imageUrl");

                            // Set as thumbnail if not set yet (use first image of first sub-workout)
                            if (!$thumbnailSet && $index === 0 && $imgIndex === 0) {
                                $mainRef->update([['path' => 'thumbnail', 'value' => $imageUrl]]);
                                $thumbnailSet = true;
                                log_message('debug', "Thumbnail set from first image of first sub-workout");
                            }
                        }
                    }
                }

                // --- Handle plan thumbnail separately (only once) ---
                if (!$thumbnailSet && isset($_FILES['planThumbnail']) && !empty($_FILES['planThumbnail']['tmp_name'])) {
                    $thumbTmp  = $_FILES['planThumbnail']['tmp_name'];
                    $thumbName = 'Thumbnails/' . uniqid() . '_' . basename($_FILES['planThumbnail']['name']);

                    $storageBucket->upload(fopen($thumbTmp, 'r'), ['name' => $thumbName]);
                    $thumbUrl = "https://firebasestorage.googleapis.com/v0/b/{$storageBucket->name()}/o/" . rawurlencode($thumbName) . "?alt=media";
                    $mainRef->update([['path' => 'thumbnail', 'value' => $thumbUrl]]);
                    $thumbnailSet = true;
                    log_message('debug', "Thumbnail set from plan thumbnail upload");
                }

                // --- Save sub workout block with multiple images ---
                $subWorkoutData = [
                    'title'       => $block['title'] ?? '',
                    'description' => $block['desc'] ?? '',
                    // Store duration in seconds per sub-workout (no conversion to minutes)
                    'duration'    => (int)($block['duration'] ?? 0),
                    'video'       => $videoUrl,
                    'images'      => $imageUrls, // NEW: Store array of image URLs
                    'createdAt'   => $nowMillis,
                    'updatedAt'   => '',
                    'order'       => $index
                ];

                log_message('debug', "Saving sub-workout $index data: " . print_r($subWorkoutData, true));

                $docRef = $mainRef->collection('subWorkOutPlan')->add($subWorkoutData);
            }

            // --- Default thumbnail if none set ---
            if (!$thumbnailSet) {
                $mainRef->update([['path' => 'thumbnail', 'value' => $defaultImageUrl]]);
            }

            echo json_encode([
                'status'  => 'success',
                'message' => 'Workout plan saved successfully',
                'planId'  => $mainDocId
            ]);
        } catch (Exception $e) {
            log_message('error', 'Workout plan save error: ' . $e->getMessage());
            log_message('error', 'Stack trace: ' . $e->getTraceAsString());

            echo json_encode([
                'status'  => 'error',
                'message' => 'Failed to save workout plan: ' . $e->getMessage()
            ]);
        }
    }

    // Helper function for JSON error responses
    private function jsonError($message, $errors = [])
    {
        echo json_encode([
            'status' => 'error',
            'message' => $message,
            'errors' => $errors
        ]);
        exit;
    }


    public function detailWorkout($id)
    {
        $plan = $this->WorkoutModel->getWorkoutPlanById($id);




        $this->load->view('pages/detail_workout_plan', ['plan' => $plan]);
    }

    public function getPlan($id)
    {
        $plan = $this->WorkoutModel->getWorkoutPlanById($id);
        $workoutTypes = $this->WorkoutModel->getWorkoutTypes();

        if (!$plan) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Plan not found'
            ]);
            return;
        }

        echo json_encode([
            'status' => 'success',
            'plan' => $plan,
            'workoutTypes' => $workoutTypes
        ]);
    }

    public function update($planId)
    {
        $post = $this->input->post();
        $files = $_FILES;

        try {
            $firestore = $this->firebase->getFirestore();
            $storage = $this->firebase->getStorage();

            $planRef = $firestore->collection('workoutplan')->document($planId);
            $snapshot = $planRef->snapshot();

            if (!$snapshot->exists()) {
                echo json_encode(['status' => 'error', 'message' => 'Plan not found']);
                return;
            }

            $title = trim($post['planTitle'] ?? '');
            if ($title === '') {
                echo json_encode(['status' => 'error', 'message' => 'Title is required']);
                return;
            }
            if (strlen($title) > 60) {
                echo json_encode(['status' => 'error', 'message' => 'Title must be 60 characters or less']);
                return;
            }

            if (!is_numeric($post['monthlyPrice'] ?? null) || ($post['monthlyPrice'] ?? 0) < 0) {
                echo json_encode(['status' => 'error', 'message' => 'Valid monthly price is required']);
                return;
            }
            if (!is_numeric($post['annualPrice'] ?? null) || ($post['annualPrice'] ?? 0) < 0) {
                echo json_encode(['status' => 'error', 'message' => 'Valid annual price is required']);
                return;
            }

            $updateData = [
                'title' => $title,
                'description' => $post['planDescription'],
                'workout_type_id' => $post['workout_type_id'],
                'monthlyPrice' => floatval($post['monthlyPrice']),
                'annualPrice' => floatval($post['annualPrice']),
                'updatedAt' => round(microtime(true) * 1000)
            ];

            // Handle thumbnail update
            if (!empty($files['planThumbnail']['tmp_name'])) {
                $thumbName = 'Thumbnails/' . uniqid() . '_' . basename($files['planThumbnail']['name']);
                $storage->upload(fopen($files['planThumbnail']['tmp_name'], 'r'), ['name' => $thumbName]);
                $thumbUrl = "https://firebasestorage.googleapis.com/v0/b/{$storage->name()}/o/" . rawurlencode($thumbName) . "?alt=media";
                $updateData['thumbnail'] = $thumbUrl;
            }

            $planRef->set($updateData, ['merge' => true]);

            echo json_encode(['status' => 'success']);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    public function delete($id)
    {
        // Validate ID
        if (empty($id)) {
            echo json_encode(['status' => 'error', 'message' => 'Invalid workout ID.']);
            return;
        }

        try {
            // Load model
            $this->load->model('WorkoutModel');

            // Attempt delete
            $deleted = $this->WorkoutModel->deleteWorkoutPlan($id);

            if ($deleted) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Workout plan deleted successfully.'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Workout plan not found or could not be deleted.'
                ]);
            }
        } catch (Exception $e) {
            log_message('error', 'Delete workout error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'An unexpected error occurred.'
            ]);
        }
    }
    public function addSubWorkout()
    {
        try {
            $planId = $this->input->post('planId');
            $title = $this->input->post('title');
            $description = $this->input->post('description') ?? '';
            $duration = (int)($this->input->post('duration') ?? 0);
            $nowMillis = round(microtime(true) * 1000);

            if (empty($planId) || empty($title)) {
                echo json_encode(['status' => 'error', 'message' => 'Missing required fields.']);
                return;
            }

            // Firebase setup
            $firestore = $this->firebase->getFirestore();
            $storageBucket = $this->firebase->getStorage();

            $defaultImageUrl = base_url('dashboard-assets/images/img/Mask.png');
            $videoUrl = '';
            $imageUrls = []; // Store multiple images

            // --- Upload Video ---
            if (isset($_FILES['video']) && is_uploaded_file($_FILES['video']['tmp_name'])) {
                $videoMime = mime_content_type($_FILES['video']['tmp_name']);
                if (strpos($videoMime, 'video/') !== 0) {
                    echo json_encode(['status' => 'error', 'message' => 'Invalid video file.']);
                    return;
                }

                $videoName = 'Videos/' . uniqid() . '_' . basename($_FILES['video']['name']);
                $storageBucket->upload(fopen($_FILES['video']['tmp_name'], 'r'), ['name' => $videoName]);
                $videoUrl = "https://firebasestorage.googleapis.com/v0/b/{$storageBucket->name()}/o/" . rawurlencode($videoName) . "?alt=media";
            }

            // --- Upload Multiple Images ---
            if (isset($_FILES['images']['tmp_name']) && is_array($_FILES['images']['tmp_name'])) {
                $imageFiles = $_FILES['images'];

                foreach ($imageFiles['tmp_name'] as $index => $tmpName) {
                    if (!empty($tmpName) && is_uploaded_file($tmpName)) {
                        $imageMime = mime_content_type($tmpName);
                        if (strpos($imageMime, 'image/') !== 0) {
                            continue; // Skip invalid images
                        }

                        $imageNameRaw = $imageFiles['name'][$index];
                        $imageName = 'SubWorkoutImages/' . uniqid() . '_' . basename($imageNameRaw);

                        // Upload to Firebase Storage
                        $storageBucket->upload(fopen($tmpName, 'r'), ['name' => $imageName]);
                        $imageUrl = "https://firebasestorage.googleapis.com/v0/b/{$storageBucket->name()}/o/" . rawurlencode($imageName) . "?alt=media";

                        $imageUrls[] = $imageUrl;
                    }
                }
            }

            // If no images were uploaded, use default placeholder
            if (empty($imageUrls)) {
                $imageUrls[] = $defaultImageUrl;
            }

            // --- Add Sub Workout Document ---
            $planRef = $firestore->collection('workoutplan')->document($planId);
            $subRef = $planRef->collection('subWorkOutPlan')->add([
                'title' => $title,
                'description' => $description,
                'duration' => $duration,
                'images' => $imageUrls, // Multiple images array
                'video' => $videoUrl,
                'createdAt' => $nowMillis,
                'updatedAt' => '',
                'order' => 0
            ]);

            // Total duration is not stored on main plan; durations are per sub-workout only

            echo json_encode([
                'status' => 'success',
                'message' => 'Sub-workout added successfully',
                'subWorkoutId' => $subRef->id(),
                'videoUrl' => $videoUrl,
                'imageUrls' => $imageUrls
            ]);
        } catch (Exception $e) {
            log_message('error', 'Add SubWorkout error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to add sub-workout: ' . $e->getMessage()
            ]);
        }
    }

    public function seedWorkoutTypes()
    {
        $firestore = $this->firebase->getFirestore();
        $collection = $firestore->collection('Workout_Type');

        // Static workout types
        $types = [
            'Full Body Workout',
            'Strength Training',
            'Cardio / HIIT',
            'Yoga / Flexibility',
            'Core / Abs',
            'Upper Body',
            'Lower Body',
            'Mobility / Recovery',
            'Pilates',
            'Special Programs'
        ];

        try {
            $nowMillis = round(microtime(true) * 1000); // Current time in milliseconds

            foreach ($types as $index => $typeName) {
                $collection->add([
                    'id' => $index + 1, // Incremental ID starting from 1
                    'name' => $typeName,
                    'description' => '', // Optional
                    'createdAt' => $nowMillis
                ]);
            }

            echo "Workout types seeded successfully with milliseconds timestamp!";
        } catch (\Exception $e) {
            log_message('error', 'Firebase Seed Error: ' . $e->getMessage());
            echo "Error seeding workout types: " . $e->getMessage();
        }
    }

    public function updateSubWorkout()
    {
        try {
            $planId = $this->input->post('planId');
            $subId = $this->input->post('subId');
            $title = $this->input->post('title');
            $description = $this->input->post('description') ?? '';
            $duration = (int)($this->input->post('duration') ?? 0);
            $nowMillis = round(microtime(true) * 1000);

            if (empty($planId) || empty($subId)) {
                echo json_encode(['status' => 'error', 'message' => 'Invalid sub-workout or plan ID']);
                return;
            }

            $firestore = $this->firebase->getFirestore();
            $storage = $this->firebase->getStorage();

            $planRef = $firestore->collection('workoutplan')->document($planId);
            $subRef = $planRef->collection('subWorkOutPlan')->document($subId);
            $snapshot = $subRef->snapshot();

            if (!$snapshot->exists()) {
                echo json_encode(['status' => 'error', 'message' => 'Sub-workout not found']);
                return;
            }

            $currentData = $snapshot->data();
            $currentImages = $currentData['images'] ?? [];

            // Handle image deletion
            $imagesToDelete = $this->input->post('imagesToDelete') ?? [];
            if (!is_array($imagesToDelete)) {
                $imagesToDelete = [];
            }

            // Delete images from Firebase Storage
            foreach ($imagesToDelete as $imageUrl) {
                try {
                    $imgPath = urldecode(parse_url($imageUrl, PHP_URL_PATH));
                    $imgPath = str_replace("/v0/b/{$storage->name()}/o/", "", $imgPath);
                    $storage->object($imgPath)->delete();
                } catch (Exception $e) {
                    log_message('error', 'Image deletion failed: ' . $e->getMessage());
                }
            }

            // Remove deleted images from current images
            $updatedImages = array_filter($currentImages, function ($image) use ($imagesToDelete) {
                return !in_array($image, $imagesToDelete);
            });

            // Handle new image uploads
            if (isset($_FILES['newImages']['tmp_name']) && is_array($_FILES['newImages']['tmp_name'])) {
                $newImageFiles = $_FILES['newImages'];

                foreach ($newImageFiles['tmp_name'] as $index => $tmpName) {
                    if (!empty($tmpName) && is_uploaded_file($tmpName)) {
                        $imageNameRaw = $newImageFiles['name'][$index];
                        $imageName = 'SubWorkoutImages/' . uniqid() . '_' . basename($imageNameRaw);

                        // Upload to Firebase Storage
                        $storage->upload(fopen($tmpName, 'r'), ['name' => $imageName]);
                        $imageUrl = "https://firebasestorage.googleapis.com/v0/b/{$storage->name()}/o/" . rawurlencode($imageName) . "?alt=media";

                        $updatedImages[] = $imageUrl;
                    }
                }
            }

            // Re-index array to ensure proper order
            $updatedImages = array_values($updatedImages);

            // Ensure at least one image exists
            if (empty($updatedImages)) {
                $updatedImages[] = base_url('dashboard-assets/images/img/Mask.png');
            }

            $updateData = [
                'title' => $title,
                'description' => $description,
                'duration' => $duration,
                'images' => $updatedImages,
                'updatedAt' => $nowMillis
            ];

            // Handle new video upload
            if (isset($_FILES['video']) && is_uploaded_file($_FILES['video']['tmp_name'])) {
                // Delete old video from storage if exists
                if (!empty($currentData['video'])) {
                    try {
                        $vidPath = urldecode(parse_url($currentData['video'], PHP_URL_PATH));
                        $vidPath = str_replace("/v0/b/{$storage->name()}/o/", "", $vidPath);
                        $storage->object($vidPath)->delete();
                    } catch (Exception $e) {
                        log_message('error', 'Video deletion failed: ' . $e->getMessage());
                    }
                }

                $videoName = 'Videos/' . uniqid() . '_' . basename($_FILES['video']['name']);
                $storage->upload(fopen($_FILES['video']['tmp_name'], 'r'), ['name' => $videoName]);
                $updateData['video'] = "https://firebasestorage.googleapis.com/v0/b/{$storage->name()}/o/" . rawurlencode($videoName) . "?alt=media";
            }

            $subRef->set($updateData, ['merge' => true]);

            // Total duration is not updated on main plan; durations remain per sub-workout

            echo json_encode(['status' => 'success', 'message' => 'Sub-workout updated successfully']);
        } catch (Exception $e) {
            log_message('error', 'Update SubWorkout error: ' . $e->getMessage());
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    public function deleteSubWorkout($planId = null, $subId = null)
    {
        try {
            if (empty($planId) || empty($subId)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Invalid plan or sub-workout ID.'
                ]);
                return;
            }

            $firestore = $this->firebase->getFirestore();
            $storage   = $this->firebase->getStorage();

            $planRef = $firestore->collection('workoutplan')->document($planId);
            $subRef  = $planRef->collection('subWorkOutPlan')->document($subId);

            $subSnap = $subRef->snapshot();

            if (!$subSnap->exists()) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Sub-workout not found.'
                ]);
                return;
            }

            $subData = $subSnap->data();

            // Delete multiple images from storage
            if (!empty($subData['images']) && is_array($subData['images'])) {
                foreach ($subData['images'] as $imageUrl) {
                    try {
                        $imgPath = urldecode(parse_url($imageUrl, PHP_URL_PATH));
                        $imgPath = str_replace("/v0/b/{$storage->name()}/o/", "", $imgPath);
                        $storage->object($imgPath)->delete();
                    } catch (Exception $e) {
                        log_message('error', 'Image deletion failed: ' . $e->getMessage());
                    }
                }
            }

            // Delete video from storage
            if (!empty($subData['video'])) {
                try {
                    $vidPath = urldecode(parse_url($subData['video'], PHP_URL_PATH));
                    $vidPath = str_replace("/v0/b/{$storage->name()}/o/", "", $vidPath);
                    $storage->object($vidPath)->delete();
                } catch (Exception $e) {
                    log_message('error', 'Video deletion failed: ' . $e->getMessage());
                }
            }

            // Delete Firestore document
            $subRef->delete();

            // Do not update main plan duration; tracked per sub-workout only

            echo json_encode([
                'status' => 'success',
                'message' => 'Sub-workout deleted successfully.',
                'remainingDuration' => $totalDuration
            ]);
        } catch (Exception $e) {
            log_message('error', 'Delete SubWorkout Error: ' . $e->getMessage());

            echo json_encode([
                'status' => 'error',
                'message' => 'Unexpected server error.'
            ]);
        }
    }
    public function library()
    {
        $page = (int)($this->input->get('page') ?? 1);
        $limit = 12;

        // Get last document ID for pagination (for next page)
        $lastDocId = $this->input->get('lastDoc');

        $result = $this->WorkoutModel->getWorkoutPlansCursor($limit, $lastDocId);

        // Store pagination history in session for "Previous" button
        $this->load->library('session');
        $paginationHistory = $this->session->userdata('library_pagination') ?? [];

        // Store current lastDocId for this page
        if ($page > 1 && !empty($lastDocId)) {
            $paginationHistory[$page] = $lastDocId;
        }

        // Get previous page's lastDocId
        $prevDocId = $paginationHistory[$page - 1] ?? null;

        $this->session->set_userdata('library_pagination', $paginationHistory);

        $this->load->view('pages/library', [
            'workoutPlans' => $result['plans'],
            'hasMore' => $result['hasMore'],
            'currentPage' => $page,
            'lastDocId' => $result['lastDocId'],
            'prevDocId' => $prevDocId
        ]);
    }

    public function libraryDetail($planId)
    {
        $plan = $this->WorkoutModel->getWorkoutPlanById($planId);

        if (!$plan) {
            redirect('dashboard/library');
            return;
        }

        $this->load->view('pages/library_detail', ['plan' => $plan]);
    }
}
