<?php
defined('BASEPATH') or exit('No direct script access allowed');

class WorkoutModel extends CI_Model
{

    protected $firebase;
    protected $firestore;
    protected $cache;
    public function __construct()
    {
        parent::__construct();

        // Get CI instance
        $CI = &get_instance();

        // Load Firebase library and cache driver
        $CI->load->library('firebase');
        $CI->load->driver('cache', array('adapter' => 'file'));

        // Assign properly
        $this->firebase  = $CI->firebase;             // the Firebase library instance
        $this->firestore = $this->firebase->getFirestore();
        $this->cache     = $CI->cache;                // cache driver instance
    }



    // 1. Fetch only workout plans (main) with sub-workout count
    public function getWorkoutPlans()
    {
        $cacheKey  = 'workout_plans_all';
        $cacheTime = 3600; // Cache for 10 minutes

        // Check cache
        $workoutPlans = $this->cache->get($cacheKey);
        if ($workoutPlans !== FALSE) {
            return $workoutPlans;
        }

        // Not in cache → fetch from Firestore
        $plansRef  = $this->firestore->collection('workoutplan');
        $snapshot  = $plansRef->documents();

        $workoutPlans = array();

        foreach ($snapshot as $doc) {
            if (!$doc->exists()) continue;

            $planData       = $doc->data();
            $planData['id'] = $doc->id();

            // Count sub-workouts
            $subWorkoutsRef = $doc->reference()->collection('subWorkOutPlan');
            $subSnapshot    = $subWorkoutsRef->documents();
            $planData['subWorkoutCount'] = iterator_count($subSnapshot);

            // Default thumbnail if missing
            $planData['thumbnail'] = isset($planData['thumbnail']) ? $planData['thumbnail'] : '';

            $workoutPlans[] = $planData;
        }

        // Save to cache
        $this->cache->save($cacheKey, $workoutPlans, $cacheTime);

        return $workoutPlans;
    }

    public function getWorkoutPlansCursor($limit = 12, $startAfterId = null)
    {
        $plansRef = $this->firestore->collection('workoutplan')->orderBy('createdAt', 'DESC');

        if ($startAfterId) {
            $docSnap = $this->firestore->collection('workoutplan')->document($startAfterId)->snapshot();
            if ($docSnap->exists()) {
                $createdAtVal = $docSnap->get('createdAt'); // int timestamp
                $plansRef = $plansRef->startAfter([$createdAtVal]);
            }
        }


        $query = $plansRef->limit($limit);
        $snapshot = $query->documents();

        $plans = [];
        $lastDocId = null;

        foreach ($snapshot as $doc) {
            if (!$doc->exists()) continue;

            $data = $doc->data();
            $data['id'] = $doc->id();

            // Count sub-workouts like getWorkoutPlans()
            $subWorkoutsRef = $doc->reference()->collection('subWorkOutPlan');
            $subSnapshot = $subWorkoutsRef->documents();
            $data['subWorkoutCount'] = iterator_count($subSnapshot);

            // Default thumbnail if missing
            $data['thumbnail'] = $data['thumbnail'] ?? '';

            // Include workout_type_id (if present)
            $data['workout_type_id'] = $data['workout_type_id'] ?? null;
            // Add workout type name
            if (!empty($data['workout_type_id'])) {
                $data['type'] = $this->getWorkoutTypeName($data['workout_type_id']);
            } else {
                $data['type'] = null;
            }


            $plans[] = $data;
            $lastDocId = $doc->id();
        }

        return [
            'plans' => $plans,
            'lastDocId' => $lastDocId,
            'hasMore' => count($plans) === $limit
        ];
    }


    public function getWorkoutPlanById($planId)
    {
        $planRef  = $this->firestore->collection('workoutplan')->document($planId);
        $planSnap = $planRef->snapshot();


        if (!$planSnap->exists()) {
            return NULL;
        }

        $planData       = $planSnap->data();
        // print_r($planData);
        // exit;
        $planData['id'] = $planSnap->id();
        $planData['thumbnail'] = isset($planData['thumbnail']) ? $planData['thumbnail'] : ''; // fallback

        if (!empty($planData['workout_type_id'])) {
            $planData['type'] = $this->getWorkoutTypeName($planData['workout_type_id']);
        } else {
            $planData['type'] = null;
        }
        // print_r($planData['type']);
        // die;
        // Fetch sub workouts
        $subSnapshot = $planRef->collection('subWorkOutPlan')->documents();
        $subWorkouts = array();
        // In your getWorkoutPlanById method, when fetching sub-workouts:
        foreach ($subSnapshot as $doc) {
            if (!$doc->exists()) continue;

            $subData = $doc->data();
            $subData['id'] = $doc->id();

            // Handle both single image (backward compatibility) and multiple images
            if (isset($subData['images']) && is_array($subData['images'])) {
                // Multiple images - use as is
                $subData['imageUrls'] = $subData['images'];
            } else {
                // Single image or legacy data
                $subData['imageUrls'] = isset($subData['image']) ? [$subData['image']] : [];
            }

            // Keep single image for backward compatibility in views
            $subData['image'] = !empty($subData['imageUrls']) ? $subData['imageUrls'][0] : '';

            $subWorkouts[] = $subData;
        }

        $planData['subWorkouts'] = $subWorkouts;

        return $planData;
    }
    public function getWorkoutTypeName($typeId)
    {
        if (empty($typeId)) {
            return null;
        }

        try {
            $typeDoc = $this->firestore->collection('Workout_Type')->document($typeId)->snapshot();

            if ($typeDoc->exists()) {
                $data = $typeDoc->data();
                return $data['name'] ?? null;
            }
        } catch (Exception $e) {
            log_message('error', 'Error fetching workout type name: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Fetch all workout types (categories) from Firestore
     *
     * @return array
     */
    public function getWorkoutTypes()
    {
        $cacheKey = 'workout_types_all';
        $cacheTime = 600; // 10 minutes cache

        // Check cache first
        $cached = $this->cache->get($cacheKey);
        if ($cached !== FALSE) {
            return $cached;
        }

        try {
            $collection = $this->firestore->collection('Workout_Type');
            $snapshot = $collection->documents();

            $types = [];
            foreach ($snapshot as $doc) {
                if (!$doc->exists()) continue;
                $data = $doc->data();
                $data['id'] = $doc->id();
                $types[] = $data;
            }

            // Save to cache
            $this->cache->save($cacheKey, $types, $cacheTime);

            return $types;
        } catch (\Exception $e) {
            log_message('error', 'Fetch Workout Types Error: ' . $e->getMessage());
            return [];
        }
    }

    // Update main workout plan fields (associative array $fields)


    // Delete main workout plan and its subWorkOutPlan docs


    public function deleteWorkoutPlan($planId)
    {
        try {
            if (empty($planId)) return false;

            $planRef = $this->firestore->collection('workoutplan')->document($planId);
            $planSnap = $planRef->snapshot();
            if (!$planSnap->exists()) return false;

            $subDocs = $planRef->collection('subWorkOutPlan')->documents();
            foreach ($subDocs as $subDoc) {
                if ($subDoc->exists()) {
                    $subDoc->reference()->delete();
                }
            }

            $planRef->delete();

            // invalidate cache
            $this->cache->delete('workout_plans_all');

            return true;
        } catch (Exception $e) {
            log_message('error', 'Model deleteWorkoutPlan error: ' . $e->getMessage());
            return false;
        }
    }
}
