<?php
// Start output buffering to capture dashboard content
ob_start();
?>

<div class="upload-form">
    <h3 class="fh2 fmon">Create Workout Plan</h3>
    <h4 class="mb-3 fs13">Workout Plans</h4>

    <!-- Upload Progress Bar -->
    <div class="progress mb-3 d-none" id="uploadProgressContainer" style="height: 25px;">
        <div class="progress-bar progress-bar-striped progress-bar-animated" id="uploadProgress" role="progressbar"
            style="width: 0%;" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100">0%</div>
    </div>

    <!-- Status Message -->
    <div id="uploadStatus" class="alert alert-info d-none">
        <div class="spinner-border spinner-border-sm me-2" role="status"></div>
        <span id="uploadStatusText">Preparing upload...</span>
    </div>

    <!-- General Form Errors -->
    <div id="generalErrors" class="alert alert-danger d-none"></div>

    <form id="workoutPlanForm" action="<?= base_url('workout/submit') ?>" method="post" enctype="multipart/form-data">
        <div class="mb-3 row">
            <div class="col-md-6">
                <label for="planTitle">Title</label>
                <input type="text" class="form-control" name="planTitle" id="planTitle" placeholder="Write title"
                    required maxlength="60">
                <small id="planTitleError" class="text-danger"></small>
            </div>
            <div class="mb-3 col-md-6">
                <label for="planDescription">Description</label>
                <textarea class="form-control" name="planDescription" id="planDescription" rows="1"
                    placeholder="Write description" required maxlength="1000"></textarea>
                <small id="planDescriptionError" class="text-danger"></small>
            </div>

            <div class="mb-3 row">
                <input type="hidden" id="billingCycle" name="billingCycle" value="both">

                <div class="mb-3 col-md-6" id="monthlyPriceGroup">
                    <label for="monthlyPrice">Monthly Price (in USD $)</label>
                    <div class="input-group">
                        <span class="input-group-text">$</span>
                        <input type="number" class="form-control" name="monthlyPrice" id="monthlyPrice"
                            placeholder="e.g., 9.99" step="0.01" min="0" max="9999.99">
                    </div>
                    <small id="monthlyPriceError" class="text-danger"></small>
                </div>

                <div class="mb-3 col-md-6" id="annualPriceGroup">
                    <label for="annualPrice">Annual Price (in USD $)</label>
                    <div class="input-group">
                        <span class="input-group-text">$</span>
                        <input type="number" class="form-control" name="annualPrice" id="annualPrice"
                            placeholder="e.g., 99.99" step="0.01" min="0" max="99999.99">
                    </div>
                    <small id="annualPriceError" class="text-danger"></small>
                </div>
                <div class="col-12">
                    <small class="text-muted">Set both monthly and annual prices; end users will choose their billing cycle at checkout.</small>
                </div>
            </div>

            <div class="col-md-6">
                <label for="planThumbnail">Workout Plan Thumbnail</label>
                <input type="file" class="form-control" name="planThumbnail" id="planThumbnail" accept="image/*">
                <small class="text-muted">Accepted formats: JPG, PNG, GIF. Max size: 5MB</small>
                <small id="planThumbnailError" class="text-danger"></small>
            </div>

            <!-- MOVED: Workout Type to main plan level -->
            <div class="mb-3 col-md-6">
                <label for="workout_type_id">Workout Type</label>
                <select class="form-control" id="workout_type_id" name="workout_type_id" required>
                    <option value="">Select</option>
                    <?php if (!empty($workoutTypes)): ?>
                        <?php foreach ($workoutTypes as $type): ?>
                            <option value="<?= htmlspecialchars($type['id']) ?>">
                                <?= htmlspecialchars($type['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </select>
                <small id="workout_type_idError" class="text-danger"></small>
            </div>
        </div>

        <!-- Sub Workout Form -->
        <div id="subWorkoutsContainer">
            <div class="mb-3 row">
                <h4 class="mb-3 fs13">Workout</h4>
                <div style="background: #ff310cf1;" class="text-white py-1 my-2 d-none" id="noSubWorkoutError">
                    <h4 class="text-white">Please add at least one workout block.</h4>
                </div>
                <div class="col-md-6">
                    <label for="workoutTitle">Workout Title</label>
                    <input type="text" class="form-control" id="workoutTitle" maxlength="255">
                    <small id="workoutTitleError" class="text-danger"></small>
                </div>
            </div>

            <div class="mb-3 row">
                <div class="col-md-6">
                    <label for="workoutDescription">Workout Details</label>
                    <textarea class="form-control" id="workoutDescription" rows="6" placeholder="Details"
                        maxlength="2000"></textarea>
                    <small id="workoutDescriptionError" class="text-danger"></small>
                </div>
                <div class="mb-3 col-md-6">
                    <label for="video">Movements</label>
                    <label style="height: 88%;" for="video"
                        class="bor1 d-flex justify-content-center align-items-center">
                        <input type="file" id="video" style="display: none;" accept="video/*">
                        <div class="d-flex justify-content-center flex-column align-items-center">
                            <img class="d-block m-auto"
                                src="<?= base_url('dashboard-assets/images/megaIcons/upload.png') ?>" alt="">
                            <p class="text-center">Tap to Upload</p>
                        </div>
                    </label>
                    <small class="text-muted">Accepted formats: MP4, MOV, AVI. Max size: 50MB</small>
                    <small id="videoError" class="text-danger"></small>
                    <div class="mt-2" id="videoPreview"></div>
                </div>
            </div>

            <!-- MULTIPLE IMAGES UPLOAD SECTION - UPDATED -->
            <div class="row">
                <div class="mb-3 col-md-6">
                    <label for="subImages">Sub Workout Images</label>
                    <input type="file" class="form-control" id="subImages" accept="image/*" multiple>
                    <small class="text-muted">Accepted formats: JPG, PNG, GIF. Max size: 5MB per image. You can select multiple images.</small>
                    <small id="subImagesError" class="text-danger"></small>

                    <!-- Multiple Images Preview Container -->
                    <div class="mt-3" id="multipleImagesPreview">
                        <div class="d-flex flex-wrap gap-2" id="selectedImagesContainer"></div>
                    </div>
                </div>
            </div>

            <div class="row">
                <div class="col-md-6 my-2">
                    <button type="button" class="btn btn-dark w-100" id="addWorkoutBtn">+ Add
                        another block of Workout</button>
                </div>
                <div class="col-md-6 my-2">
                    <button type="submit" class="btn btn-dark w-100" id="submitBtn">Share Workout Plan</button>
                </div>
            </div>
        </div>
    </form>

    <!-- Sub Workout Blocks Table -->
    <h4 class="mb-3 fs13">Workout Blocks</h4>
    <table class="table table-bordered" id="subWorkoutTable">
        <thead>
            <tr>
                <th>Images</th>
                <th>Title</th>
                <th>Duration</th>
                <th>Description</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody></tbody>
    </table>
</div>

<style>
    /* Multiple Images Preview Styles */
    .image-preview-item {
        position: relative;
        width: 100px;
        height: 100px;
        border: 2px solid #ddd;
        border-radius: 8px;
        overflow: hidden;
        cursor: move;
    }

    .image-preview-item img {
        width: 100%;
        height: 100%;
        object-fit: cover;
    }

    .image-preview-item .remove-btn {
        position: absolute;
        top: 2px;
        right: 2px;
        background: rgba(255, 0, 0, 0.7);
        color: white;
        border: none;
        border-radius: 50%;
        width: 20px;
        height: 20px;
        font-size: 12px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .image-preview-item .remove-btn:hover {
        background: rgba(255, 0, 0, 0.9);
    }

    .image-preview-item.dragging {
        opacity: 0.5;
    }

    #selectedImagesContainer .sortable-ghost {
        opacity: 0.4;
    }

    .drag-handle {
        position: absolute;
        top: 2px;
        left: 2px;
        background: rgba(0, 0, 0, 0.7);
        color: white;
        border: none;
        border-radius: 50%;
        width: 20px;
        height: 20px;
        font-size: 10px;
        cursor: move;
        display: flex;
        align-items: center;
        justify-content: center;
    }
</style>

<script>
    document.addEventListener("DOMContentLoaded", function() {
        let subWorkouts = [];
        let selectedImages = []; // NEW: Store multiple images for current sub-workout

        // Helper function to get element by ID
        function getEl(id) {
            return document.getElementById(id);
        }

        // Function to show general errors
        function showGeneralError(message) {
            const errorDiv = getEl("generalErrors");
            errorDiv.innerHTML = `<strong>Error:</strong> ${message}`;
            errorDiv.classList.remove("d-none");
        }

        // Function to hide general errors
        function hideGeneralError() {
            getEl("generalErrors").classList.add("d-none");
        }

        // Function to update progress bar
        function updateProgressBar(percentage) {
            const progressContainer = getEl("uploadProgressContainer");
            const progressBar = getEl("uploadProgress");

            progressContainer.classList.remove("d-none");
            progressBar.style.width = percentage + "%";
            progressBar.setAttribute("aria-valuenow", percentage);
            progressBar.textContent = percentage + "%";
        }

        // Function to show upload status
        function showUploadStatus(message) {
            const statusElement = getEl("uploadStatus");
            const statusText = getEl("uploadStatusText");

            statusText.textContent = message;
            statusElement.classList.remove("d-none");
        }

        // Function to hide upload status
        function hideUploadStatus() {
            getEl("uploadStatus").classList.add("d-none");
        }

        // Function to get video duration
        function getVideoDuration(file) {
            return new Promise((resolve, reject) => {
                const video = document.createElement('video');
                video.preload = 'metadata';

                video.onloadedmetadata = function() {
                    window.URL.revokeObjectURL(video.src);
                    const duration = Math.round(video.duration); // Duration in seconds
                    resolve(duration);
                };

                video.onerror = function() {
                    reject(new Error('Failed to load video metadata'));
                };

                video.src = URL.createObjectURL(file);
            });
        }

        // Function to format duration (seconds to mm:ss)
        function formatDuration(seconds) {
            const mins = Math.floor(seconds / 60);
            const secs = seconds % 60;
            return `${mins}:${secs.toString().padStart(2, '0')}`;
        }

        // SweetAlert2 helper functions
        function showSuccess(message) {
            Swal.fire({
                icon: "success",
                title: "Success",
                text: message,
                timer: 2000,
                showConfirmButton: false,
            });
        }

        function showError(message) {
            Swal.fire({
                icon: "error",
                title: "Error",
                text: message,
            });
        }

        function showLoading(message = "Uploading...") {
            Swal.fire({
                title: message,
                didOpen: () => Swal.showLoading(),
                allowOutsideClick: false,
                allowEscapeKey: false,
                showConfirmButton: false,
            });
        }

        function closePopup() {
            Swal.close();
        }

        // Reset all error states
        function resetAllErrors() {
            // Clear error messages
            const errorElements = document.querySelectorAll(".text-danger");
            errorElements.forEach((el) => (el.textContent = ""));

            // Remove invalid classes
            const invalidElements = document.querySelectorAll(".is-invalid");
            invalidElements.forEach((el) => el.classList.remove("is-invalid"));

            // Hide general errors
            hideGeneralError();
            getEl("noSubWorkoutError").classList.add("d-none");
        }

        // Validate file function
        function validateFile(file, allowedTypes, maxSize, fieldName) {
            if (!file) return null;

            const fileType = file.type;
            const fileSize = file.size;
            const maxSizeBytes = maxSize * 1024 * 1024;

            if (!allowedTypes.some(type => fileType.includes(type))) {
                return `Invalid file type for ${fieldName}. Allowed types: ${allowedTypes.join(', ')}`;
            }

            if (fileSize > maxSizeBytes) {
                return `${fieldName} must be less than ${maxSize}MB`;
            }

            return null;
        }

        // NEW: Multiple Images Handler with Drag & Drop
        getEl("subImages").addEventListener("change", function(e) {
            const previewContainer = getEl("selectedImagesContainer");
            getEl("subImagesError").textContent = "";

            if (this.files && this.files.length > 0) {
                const files = Array.from(this.files);

                // Validate each file
                for (let file of files) {
                    const error = validateFile(file, ['jpeg', 'jpg', 'png', 'gif'], 5, "Image");
                    if (error) {
                        getEl("subImagesError").textContent = error;
                        continue;
                    }

                    // Check for duplicate files
                    if (!selectedImages.some(img => img.name === file.name && img.size === file.size)) {
                        selectedImages.push(file);
                        addImagePreview(file, selectedImages.length - 1);
                    }
                }

                // Reset file input to allow selecting same files again
                this.value = "";

                updateImageOrder();
            }
        });

        // NEW: Function to add image preview
        function addImagePreview(file, index) {
            const previewContainer = getEl("selectedImagesContainer");

            const previewItem = document.createElement("div");
            previewItem.className = "image-preview-item";
            previewItem.setAttribute("data-index", index);
            previewItem.draggable = true;

            const img = document.createElement("img");
            img.src = URL.createObjectURL(file);

            const removeBtn = document.createElement("button");
            removeBtn.type = "button";
            removeBtn.className = "remove-btn";
            removeBtn.innerHTML = "×";
            removeBtn.onclick = function() {
                removeImage(index);
            };

            const dragHandle = document.createElement("div");
            dragHandle.className = "drag-handle";
            dragHandle.innerHTML = "≡";

            previewItem.appendChild(img);
            previewItem.appendChild(removeBtn);
            previewItem.appendChild(dragHandle);
            previewContainer.appendChild(previewItem);

            // Add drag and drop events
            setupDragAndDrop(previewItem);
        }

        // NEW: Function to remove image
        function removeImage(index) {
            selectedImages.splice(index, 1);
            refreshImagePreviews();
        }

        // NEW: Function to refresh all image previews
        function refreshImagePreviews() {
            const previewContainer = getEl("selectedImagesContainer");
            previewContainer.innerHTML = "";

            selectedImages.forEach((file, index) => {
                addImagePreview(file, index);
            });
        }

        // NEW: Function to update image order display
        function updateImageOrder() {
            const previewItems = document.querySelectorAll('.image-preview-item');
            previewItems.forEach((item, index) => {
                item.setAttribute('data-index', index);
            });
        }

        // NEW: Drag and Drop setup
        function setupDragAndDrop(item) {
            item.addEventListener('dragstart', function(e) {
                e.dataTransfer.setData('text/plain', this.getAttribute('data-index'));
                this.classList.add('dragging');
            });

            item.addEventListener('dragend', function(e) {
                this.classList.remove('dragging');
            });

            item.addEventListener('dragover', function(e) {
                e.preventDefault();
            });

            item.addEventListener('drop', function(e) {
                e.preventDefault();
                const fromIndex = parseInt(e.dataTransfer.getData('text/plain'));
                const toIndex = parseInt(this.getAttribute('data-index'));

                if (fromIndex !== toIndex) {
                    // Reorder the images array
                    const [movedItem] = selectedImages.splice(fromIndex, 1);
                    selectedImages.splice(toIndex, 0, movedItem);

                    refreshImagePreviews();
                }
            });
        }

        // NEW: Initialize drag and drop on container
        getEl("selectedImagesContainer").addEventListener('dragover', function(e) {
            e.preventDefault();
        });

        getEl("selectedImagesContainer").addEventListener('drop', function(e) {
            e.preventDefault();
        });

        // Video preview handler with enhanced validation and duration capture
        getEl("video").addEventListener("change", async function(e) {
            const preview = getEl("videoPreview");
            preview.innerHTML = "";
            getEl("videoError").textContent = "";

            if (this.files && this.files[0]) {
                const file = this.files[0];
                const error = validateFile(file, ['mp4', 'mov', 'avi'], 50, "Video");

                if (error) {
                    getEl("videoError").textContent = error;
                    this.value = "";
                    return;
                }

                try {
                    // Get video duration
                    const duration = await getVideoDuration(file);

                    const fileName = document.createElement("p");
                    fileName.textContent = `Selected: ${file.name} (${(file.size / (1024 * 1024)).toFixed(2)} MB) - Duration: ${formatDuration(duration)}`;
                    fileName.className = "text-success small";
                    preview.appendChild(fileName);

                    // Store duration in a data attribute for later use
                    this.setAttribute('data-duration', duration);
                } catch (err) {
                    console.error('Error getting video duration:', err);
                    const fileName = document.createElement("p");
                    fileName.textContent = `Selected: ${file.name} (${(file.size / (1024 * 1024)).toFixed(2)} MB)`;
                    fileName.className = "text-success small";
                    preview.appendChild(fileName);

                    // Set default duration if unable to get it
                    this.setAttribute('data-duration', '0');
                }
            }
        });

        // Plan thumbnail validation
        getEl("planThumbnail").addEventListener("change", function(e) {
            getEl("planThumbnailError").textContent = "";

            if (this.files && this.files[0]) {
                const file = this.files[0];
                const error = validateFile(file, ['jpeg', 'jpg', 'png', 'gif'], 5, "Thumbnail");

                if (error) {
                    getEl("planThumbnailError").textContent = error;
                    this.value = "";
                }
            }
        });

        // UPDATED: Add sub-workout function with multiple images support
        function addSubWorkout() {
            resetAllErrors();

            const title = getEl("workoutTitle").value.trim();
            const desc = getEl("workoutDescription").value.trim();
            const videoInput = getEl("video");
            const videoFile = videoInput.files[0];
            const videoDuration = parseInt(videoInput.getAttribute('data-duration') || '0');

            // Validation
            let isValid = true;

            if (!title) {
                getEl("workoutTitleError").textContent = "Workout title is required.";
                getEl("workoutTitle").classList.add("is-invalid");
                isValid = false;
            } else if (title.length > 255) {
                getEl("workoutTitleError").textContent = "Workout title must be less than 255 characters.";
                getEl("workoutTitle").classList.add("is-invalid");
                isValid = false;
            }

            if (!desc) {
                getEl("workoutDescriptionError").textContent = "Workout details are required.";
                getEl("workoutDescription").classList.add("is-invalid");
                isValid = false;
            } else if (desc.length > 2000) {
                getEl("workoutDescriptionError").textContent = "Workout details must be less than 2000 characters.";
                getEl("workoutDescription").classList.add("is-invalid");
                isValid = false;
            }

            // UPDATED: Either video OR at least one image is required
            if (!videoFile && selectedImages.length === 0) {
                getEl("videoError").textContent = "Either video or at least one image is required.";
                videoInput.classList.add("is-invalid");
                isValid = false;
            }

            if (videoFile) {
                const videoError = validateFile(videoFile, ['mp4', 'mov', 'avi'], 50, "Video");
                if (videoError) {
                    getEl("videoError").textContent = videoError;
                    videoInput.classList.add("is-invalid");
                    isValid = false;
                }
            }

            // UPDATED: Validate each image if any are selected
            if (selectedImages.length > 0) {
                selectedImages.forEach((file, index) => {
                    const imageError = validateFile(file, ['jpeg', 'jpg', 'png', 'gif'], 5, "Image");
                    if (imageError) {
                        getEl("subImagesError").textContent = `Image ${index + 1}: ${imageError}`;
                        isValid = false;
                    }
                });
            }

            if (!isValid) {
                showError("Please fix the errors in the workout block before adding.");
                return;
            }

            // Check for duplicate titles in current session
            const duplicate = subWorkouts.find(sw => sw.title.toLowerCase() === title.toLowerCase());
            if (duplicate) {
                getEl("workoutTitleError").textContent = "A workout with this title already exists.";
                getEl("workoutTitle").classList.add("is-invalid");
                showError("A workout with this title already exists. Please use a different title.");
                return;
            }

            // UPDATED: Store multiple images array instead of single image
            subWorkouts.push({
                title,
                desc,
                video: videoFile,
                duration: videoDuration,
                images: [...selectedImages], // Store array of images
            });
            console.log("Current subWorkouts:", subWorkouts);

            // UPDATED: Image preview in table - show first image + count
            let imagePreview = "";
            if (selectedImages.length > 0) {
                const firstImageUrl = URL.createObjectURL(selectedImages[0]);
                imagePreview = `<img src="${firstImageUrl}" alt="Preview" style="width: 60px; height: 60px; object-fit: cover; border-radius: 5px;">`;
                if (selectedImages.length > 1) {
                    imagePreview += `<small class="text-muted d-block">+${selectedImages.length - 1} more</small>`;
                }
            } else {
                imagePreview = `<span class="text-muted">No Images</span>`;
            }

            const tbody = document.querySelector("#subWorkoutTable tbody");
            const row = document.createElement("tr");
            row.innerHTML = `
                <td>${imagePreview}</td>
                <td>${title}</td>
                <td>${formatDuration(videoDuration)}</td>
                <td>${desc.substring(0, 50)}${desc.length > 50 ? "..." : ""}</td>
                <td><button type="button" class="btn btn-sm btn-danger" onclick="removeSubWorkout(this)">Remove</button></td>
            `;
            tbody.appendChild(row);

            // Reset input fields
            getEl("workoutTitle").value = "";
            getEl("workoutDescription").value = "";
            videoInput.value = "";
            videoInput.removeAttribute('data-duration');
            getEl("subImages").value = "";
            getEl("videoPreview").innerHTML = "";

            // UPDATED: Clear multiple images
            selectedImages = [];
            getEl("selectedImagesContainer").innerHTML = "";

            showSuccess("Workout block added successfully!");
        }

        // Remove sub-workout function
        window.removeSubWorkout = function(btn) {
            const row = btn.closest("tr");
            const index = Array.from(row.parentNode.children).indexOf(row);
            const removedWorkout = subWorkouts[index];

            subWorkouts.splice(index, 1);
            row.remove();

            showSuccess(`"${removedWorkout.title}" removed successfully!`);
        };

        // Billing is fixed to "both" – always show and require both prices
        const billingCycleEl = getEl("billingCycle");
        function showBothPrices() {
            getEl("monthlyPriceGroup").style.display = "block";
            getEl("annualPriceGroup").style.display = "block";
            getEl("monthlyPrice").required = true;
            getEl("annualPrice").required = true;
        }
        showBothPrices();

        // Enhanced form validation
        function validateForm() {
            resetAllErrors();

            let isValid = true;
            const title = getEl("planTitle").value.trim();
            const description = getEl("planDescription").value.trim();
            const billingCycle = "both";
            const workoutTypeId = getEl("workout_type_id").value;
            const thumbnailFile = getEl("planThumbnail").files[0];

            // Title validation
            if (!title) {
                getEl("planTitleError").textContent = "Title is required.";
                getEl("planTitle").classList.add("is-invalid");
                isValid = false;
            } else if (title.length > 60) {
                getEl("planTitleError").textContent = "Title must be 60 characters or less.";
                getEl("planTitle").classList.add("is-invalid");
                isValid = false;
            }

            // Description validation
            if (!description) {
                getEl("planDescriptionError").textContent = "Description is required.";
                getEl("planDescription").classList.add("is-invalid");
                isValid = false;
            } else if (description.length > 1000) {
                getEl("planDescriptionError").textContent = "Description must be less than 1000 characters.";
                getEl("planDescription").classList.add("is-invalid");
                isValid = false;
            }

            // Workout type validation
            if (!workoutTypeId) {
                getEl("workout_type_idError").textContent = "Workout type is required.";
                getEl("workout_type_id").classList.add("is-invalid");
                isValid = false;
            }

            // Pricing validation (always require both; user picks later)
            const monthlyPrice = getEl("monthlyPrice").value.trim();
            const annualPrice = getEl("annualPrice").value.trim();

            if (!monthlyPrice || isNaN(monthlyPrice) || Number(monthlyPrice) < 0) {
                getEl("monthlyPriceError").textContent = "Valid monthly price is required.";
                getEl("monthlyPrice").classList.add("is-invalid");
                isValid = false;
            } else if (Number(monthlyPrice) > 9999.99) {
                getEl("monthlyPriceError").textContent = "Monthly price cannot exceed $9999.99.";
                getEl("monthlyPrice").classList.add("is-invalid");
                isValid = false;
            }

            if (!annualPrice || isNaN(annualPrice) || Number(annualPrice) < 0) {
                getEl("annualPriceError").textContent = "Valid annual price is required.";
                getEl("annualPrice").classList.add("is-invalid");
                isValid = false;
            } else if (Number(annualPrice) > 99999.99) {
                getEl("annualPriceError").textContent = "Annual price cannot exceed $99999.99.";
                getEl("annualPrice").classList.add("is-invalid");
                isValid = false;
            }

            if (isValid && Number(annualPrice) <= Number(monthlyPrice)) {
                getEl("annualPriceError").textContent = "Annual price should be greater than monthly price.";
                getEl("annualPrice").classList.add("is-invalid");
                isValid = false;
            }

            // Thumbnail validation (optional but if provided, validate)
            if (thumbnailFile) {
                const thumbnailError = validateFile(thumbnailFile, ['jpeg', 'jpg', 'png', 'gif'], 5, "Thumbnail");
                if (thumbnailError) {
                    getEl("planThumbnailError").textContent = thumbnailError;
                    getEl("planThumbnail").classList.add("is-invalid");
                    isValid = false;
                }
            }

            // Sub workouts validation
            if (subWorkouts.length === 0) {
                getEl("noSubWorkoutError").classList.remove("d-none");
                showGeneralError("Please add at least one workout block.");
                isValid = false;
            }

            return isValid;
        }

        // UPDATED: Submit handler with multiple images support
        getEl("workoutPlanForm").addEventListener("submit", function(e) {
            e.preventDefault();

            console.log("Form submitted - SubWorkouts count:", subWorkouts.length);

            if (!validateForm()) {
                const firstError = document.querySelector(".is-invalid");
                if (firstError) {
                    firstError.scrollIntoView({
                        behavior: 'smooth',
                        block: 'center'
                    });
                }
                return;
            }

            // Get the current values from the form elements
            const title = getEl("planTitle").value.trim();
            const description = getEl("planDescription").value.trim();
            const billingCycle = "both";
            const monthlyPrice = getEl("monthlyPrice").value.trim();
            const annualPrice = getEl("annualPrice").value.trim();
            const workoutTypeId = getEl("workout_type_id").value;

            // ✅ USE THE FORM ELEMENT TO CREATE FormData
            const form = this;
            const formData = new FormData(form);

            // ✅ Add/override fields with current values
            formData.set("planTitle", title);
            formData.set("planDescription", description);
            formData.set("billingCycle", billingCycle);
            formData.set("workout_type_id", workoutTypeId);
            formData.set("monthlyPrice", monthlyPrice);
            formData.set("annualPrice", annualPrice);

            // ✅ UPDATED: Prepare and add subWorkouts with multiple images support
            const cleanSubWorkouts = subWorkouts.map((sw, index) => ({
                title: sw.title,
                desc: sw.desc,
                duration: sw.duration || 0,
                videoName: sw.video?.name ?? null,
                imageNames: sw.images ? sw.images.map(img => img.name) : [], // Array of image names
                imageCount: sw.images ? sw.images.length : 0
            }));

            console.log("Clean SubWorkouts to send:", cleanSubWorkouts);

            cleanSubWorkouts.forEach((sw, index) => {
                formData.append(`subWorkouts[${index}][title]`, sw.title);
                formData.append(`subWorkouts[${index}][desc]`, sw.desc);
                formData.append(`subWorkouts[${index}][duration]`, sw.duration);
                formData.append(`subWorkouts[${index}][videoName]`, sw.videoName);
                formData.append(`subWorkouts[${index}][imageCount]`, sw.imageCount);

                // Add individual image names
                sw.imageNames.forEach((imgName, imgIndex) => {
                    formData.append(`subWorkouts[${index}][imageNames][${imgIndex}]`, imgName);
                });
            });

            // ✅ UPDATED: Add files with multiple images support
            subWorkouts.forEach((sw, index) => {
                if (sw.video) {
                    formData.append(`videos[]`, sw.video);
                }
                if (sw.images && sw.images.length > 0) {
                    sw.images.forEach((image, imgIndex) => {
                        formData.append(`subWorkoutImages[${index}][]`, image);
                    });
                }
            });

            // Add plan thumbnail if exists
            const planThumbnail = getEl("planThumbnail").files[0];
            if (planThumbnail) {
                formData.append("planThumbnail", planThumbnail);
            }

            // DEBUG: Check what's in FormData
            console.log("=== FORM DATA DEBUG ===");
            for (let pair of formData.entries()) {
                if (pair[1] instanceof File) {
                    console.log(pair[0] + ':', pair[1].name, '(' + pair[1].size + ' bytes)');
                } else {
                    console.log(pair[0] + ':', pair[1]);
                }
            }

            // Show loading state
            showLoading("Uploading your workout plan...");
            showUploadStatus("Starting upload...");
            updateProgressBar(0);

            getEl("addWorkoutBtn").disabled = true;
            getEl("submitBtn").disabled = true;

            const xhr = new XMLHttpRequest();
            xhr.open("POST", form.action, true);
            xhr.timeout = 300000; // 5 minutes timeout

            xhr.upload.addEventListener("progress", function(evt) {
                if (evt.lengthComputable) {
                    const percentComplete = Math.round((evt.loaded / evt.total) * 100);
                    updateProgressBar(percentComplete);
                    showUploadStatus(`Uploading... ${percentComplete}% complete`);
                }
            });

            xhr.onload = function() {
                closePopup();
                hideUploadStatus();

                console.log("Response status:", xhr.status);
                console.log("Response text:", xhr.responseText);

                if (xhr.status >= 200 && xhr.status < 300) {
                    try {
                        const result = JSON.parse(xhr.responseText);
                        console.log("Server response:", result);

                        if (result.status === "success") {
                            showSuccess(result.message || "Workout plan created successfully!");
                            setTimeout(() => {
                                window.location.href = "<?= base_url('dashboard/workout-plans') ?>";
                            }, 2000);
                        } else if (result.status === "error") {
                            let errorMessage = result.message || "Something went wrong";

                            // Handle backend validation errors
                            if (result.errors) {
                                resetAllErrors();

                                let firstErrorField = null;
                                for (const field in result.errors) {
                                    if (result.errors.hasOwnProperty(field)) {
                                        const fieldEl = getEl(field);
                                        const errorEl = getEl(field + "Error");

                                        if (fieldEl) {
                                            fieldEl.classList.add("is-invalid");
                                            if (!firstErrorField) firstErrorField = fieldEl;
                                        }
                                        if (errorEl) {
                                            errorEl.textContent = Array.isArray(result.errors[field]) ?
                                                result.errors[field].join(', ') :
                                                result.errors[field];
                                        }
                                    }
                                }

                                // Scroll to first error
                                if (firstErrorField) {
                                    firstErrorField.scrollIntoView({
                                        behavior: 'smooth',
                                        block: 'center'
                                    });
                                }

                                errorMessage = "Please fix the errors highlighted below.";
                            }

                            showError(errorMessage);
                        }
                    } catch (e) {
                        console.error("JSON parse error:", e);
                        console.error("Raw response:", xhr.responseText);
                        showError("Invalid response from server. Please try again.");
                    }
                } else {
                    handleHttpError(xhr.status);
                }

                getEl("addWorkoutBtn").disabled = false;
                getEl("submitBtn").disabled = false;
            };

            xhr.onerror = function() {
                closePopup();
                hideUploadStatus();
                showError("Network error. Please check your connection and try again.");
                console.error("XHR error");

                getEl("addWorkoutBtn").disabled = false;
                getEl("submitBtn").disabled = false;
            };

            xhr.ontimeout = function() {
                closePopup();
                hideUploadStatus();
                showError("Request timeout. Please try again with smaller files or better connection.");

                getEl("addWorkoutBtn").disabled = false;
                getEl("submitBtn").disabled = false;
            };

            xhr.send(formData);
        });

        // Handle HTTP errors
        function handleHttpError(status) {
            switch (status) {
                case 413:
                    showError("File too large. Please ensure your files are within size limits.");
                    break;
                case 415:
                    showError("Unsupported file type. Please use accepted formats.");
                    break;
                case 500:
                    showError("Server error. Please try again later.");
                    break;
                case 403:
                    showError("Permission denied. Please check your account permissions.");
                    break;
                case 401:
                    showError("Session expired. Please refresh the page and login again.");
                    break;
                default:
                    showError(`Request failed with status ${status}. Please try again.`);
            }
        }

        // Bind events
        getEl("addWorkoutBtn").addEventListener("click", addSubWorkout);

        // Add input event listeners for real-time validation
        const realTimeValidationFields = [
            'planTitle', 'planDescription', 'monthlyPrice', 'annualPrice',
            'workoutTitle', 'workoutDescription', 'workout_type_id'
        ];

        realTimeValidationFields.forEach(fieldId => {
            const field = getEl(fieldId);
            if (field) {
                field.addEventListener('input', function() {
                    if (this.classList.contains('is-invalid')) {
                        const errorEl = getEl(fieldId + 'Error');
                        if (errorEl) errorEl.textContent = '';
                        this.classList.remove('is-invalid');
                    }
                });
            }
        });
    });
</script>

<?php
// Capture all content in a variable
$content = ob_get_clean();

// Load the layout and pass the content
$this->load->view('layouts/app', ['content' => $content]);
?>
