<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>;
// Wait for DOM to be fully loaded
document.addEventListener("DOMContentLoaded", function () {
	let subWorkouts = [];

	// Helper function to get element by ID
	function getEl(id) {
		return document.getElementById(id);
	}

	// Function to update progress bar
	function updateProgressBar(percentage) {
		const progressContainer = getEl("uploadProgressContainer");
		const progressBar = getEl("uploadProgress");

		progressContainer.classList.remove("d-none");
		progressBar.style.width = percentage + "%";
		progressBar.setAttribute("aria-valuenow", percentage);
		progressBar.textContent = percentage + "%";
	}

	// Function to show upload status
	function showUploadStatus(message) {
		const statusElement = getEl("uploadStatus");
		const statusText = getEl("uploadStatusText");

		statusText.textContent = message;
		statusElement.classList.remove("d-none");
	}

	// Function to hide upload status
	function hideUploadStatus() {
		getEl("uploadStatus").classList.add("d-none");
	}

	// SweetAlert2 helper functions
	function showSuccess(message) {
		Swal.fire({
			icon: "success",
			title: "Success",
			text: message,
			timer: 2000,
			showConfirmButton: false,
		});
	}

	function showError(message) {
		Swal.fire({
			icon: "error",
			title: "Error",
			text: message,
		});
	}

	function showLoading(message = "Uploading...") {
		Swal.fire({
			title: message,
			didOpen: () => Swal.showLoading(),
			allowOutsideClick: false,
			allowEscapeKey: false,
			showConfirmButton: false,
		});
	}

	function closePopup() {
		Swal.close();
	}

	// Video preview handler
	getEl("video").addEventListener("change", function (e) {
		const preview = getEl("videoPreview");
		preview.innerHTML = "";

		if (this.files && this.files[0]) {
			const file = this.files[0];

			if (file.size > 50 * 1024 * 1024) {
				showError("Video file must be less than 50MB");
				this.value = "";
				return;
			}

			const fileName = document.createElement("p");
			fileName.textContent = `Selected: ${file.name} (${(
				file.size /
				(1024 * 1024)
			).toFixed(2)} MB)`;
			fileName.className = "text-success small";
			preview.appendChild(fileName);
		}
	});

	// Image preview handler
	getEl("subImage").addEventListener("change", function (e) {
		const preview = getEl("imagePreview");
		preview.innerHTML = "";

		if (this.files && this.files[0]) {
			const file = this.files[0];

			if (!file.type.match("image.*")) {
				showError("Please select an image file");
				this.value = "";
				return;
			}

			const fileName = document.createElement("p");
			fileName.textContent = `Selected: ${file.name}`;
			fileName.className = "text-success small";
			preview.appendChild(fileName);

			const img = document.createElement("img");
			img.src = URL.createObjectURL(file);
			img.style.maxWidth = "100px";
			img.style.maxHeight = "100px";
			img.className = "mt-2";
			preview.appendChild(img);
		}
	});

	// Add sub-workout function
	function addSubWorkout() {
		const title = getEl("workoutTitle").value.trim();
		const type = getEl("category").value;
		const desc = getEl("workoutDescription").value.trim();
		const videoInput = getEl("video");
		const videoFile = videoInput.files[0];
		const imageInput = getEl("subImage");
		const imageFile = imageInput.files[0];

		// Reset errors
		getEl("workoutTitleError").textContent = "";
		getEl("categoryError").textContent = "";
		getEl("workoutDescriptionError").textContent = "";
		getEl("videoError").textContent = "";

		// Validation
		let isValid = true;
		if (!title) {
			getEl("workoutTitleError").textContent = "Workout title is required.";
			isValid = false;
		}
		if (!type) {
			getEl("categoryError").textContent = "Workout type is required.";
			isValid = false;
		}
		if (!desc) {
			getEl("workoutDescriptionError").textContent =
				"Workout details are required.";
			isValid = false;
		}
		if (!videoFile) {
			getEl("videoError").textContent = "Video is required.";
			isValid = false;
		}

		if (!isValid) return;

		subWorkouts.push({
			title,
			type,
			desc,
			video: videoFile,
			image: imageFile || null,
		});

		// Image preview
		let imagePreview = "";
		if (imageFile) {
			const imageUrl = URL.createObjectURL(imageFile);
			imagePreview = `<img src="${imageUrl}" alt="Preview" style="width: 60px; height: 60px; object-fit: cover; border-radius: 5px;">`;
		} else {
			imagePreview = `<span class="text-muted">No Image</span>`;
		}

		const tbody = document.querySelector("#subWorkoutTable tbody");
		const row = document.createElement("tr");
		row.innerHTML = `
                <td>${imagePreview}</td>
                <td>${title}</td>
                <td>${type}</td>
                <td>${desc.substring(0, 50)}${
			desc.length > 50 ? "..." : ""
		}</td>
                <td><button type="button" class="btn btn-sm btn-danger" onclick="removeSubWorkout(this)">Remove</button></td>
            `;
		tbody.appendChild(row);

		// Reset input fields
		getEl("workoutTitle").value = "";
		getEl("category").value = "";
		getEl("workoutDescription").value = "";
		videoInput.value = "";
		imageInput.value = "";
		getEl("videoPreview").innerHTML = "";
		getEl("imagePreview").innerHTML = "";
	}

	// Remove sub-workout function
	window.removeSubWorkout = function (btn) {
		const row = btn.closest("tr");
		const index = Array.from(row.parentNode.children).indexOf(row);
		subWorkouts.splice(index, 1);
		row.remove();
	};

	// Submit handler with SweetAlert upload
	getEl("workoutPlanForm").addEventListener("submit", function (e) {
		e.preventDefault();

		const errorElements = document.querySelectorAll(".text-danger");
		errorElements.forEach((el) => (el.textContent = ""));

		const invalidElements = document.querySelectorAll(".is-invalid");
		invalidElements.forEach((el) => el.classList.remove("is-invalid"));

		const errorDiv = getEl("noSubWorkoutError");
		if (subWorkouts.length === 0) {
			errorDiv.classList.remove("d-none");
			showError("Please add at least one workout block.");
			return;
		} else {
			errorDiv.classList.add("d-none");
		}

		let isValid = true;
		const title = getEl("planTitle").value.trim();
		const description = getEl("planDescription").value.trim();
		const monthlyPrice = getEl("monthlyPrice").value.trim();
		const annualPrice = getEl("annualPrice").value.trim();

		if (!title) {
			getEl("planTitleError").textContent = "Title is required.";
			getEl("planTitle").classList.add("is-invalid");
			isValid = false;
		}

		if (!description) {
			getEl("planDescriptionError").textContent = "Description is required.";
			getEl("planDescription").classList.add("is-invalid");
			isValid = false;
		}

		if (!monthlyPrice || isNaN(monthlyPrice) || Number(monthlyPrice) < 0) {
			getEl("monthlyPriceError").textContent =
				"Valid monthly price is required.";
			getEl("monthlyPrice").classList.add("is-invalid");
			isValid = false;
		}

		if (!annualPrice || isNaN(annualPrice) || Number(annualPrice) < 0) {
			getEl("annualPriceError").textContent =
				"Valid annual price is required.";
			getEl("annualPrice").classList.add("is-invalid");
			isValid = false;
		}

		if (!isValid) return;

		const form = this;
		const formData = new FormData(form);

		formData.append("planTitle", title);
		formData.append("planDescription", description);
		formData.append("monthlyPrice", monthlyPrice);
		formData.append("annualPrice", annualPrice);

		const cleanSubWorkouts = subWorkouts.map((sw) => ({
			title: sw.title,
			type: sw.type,
			desc: sw.desc,
			videoName: sw.video?.name ?? null,
			imageName: sw.image?.name ?? null,
		}));

		formData.append("subWorkouts", JSON.stringify(cleanSubWorkouts));

		subWorkouts.forEach((sw) => {
			formData.append("videos[]", sw.video);
			formData.append("images[]", sw.image || new File([], "empty"));
		});

		// **SweetAlert loading modal**
		showLoading("Uploading your workout plan...");
		showUploadStatus("Starting upload...");
		updateProgressBar(0);

		getEl("addWorkoutBtn").disabled = true;
		getEl("submitBtn").disabled = true;

		const xhr = new XMLHttpRequest();
		xhr.open("POST", form.action, true);

		xhr.upload.addEventListener("progress", function (evt) {
			if (evt.lengthComputable) {
				const percentComplete = Math.round((evt.loaded / evt.total) * 100);
				updateProgressBar(percentComplete);
				showUploadStatus(`Uploading... ${percentComplete}% complete`);
			}
		});

		xhr.onload = function () {
			closePopup();
			hideUploadStatus();

			if (xhr.status >= 200 && xhr.status < 300) {
				try {
					const result = JSON.parse(xhr.responseText);

					if (result.status === "success") {
						showSuccess(result.message || "Workout uploaded successfully");
						setTimeout(() => {
							window.location.href =
								"<?= base_url('dashboard/workout-plans') ?>";
						}, 2000);
					} else if (result.status === "error") {
						showError(result.message || "Something went wrong");
						if (result.errors) {
							for (const field in result.errors) {
								if (result.errors.hasOwnProperty(field)) {
									const fieldEl = getEl(field);
									const errorEl = getEl(field + "Error");
									if (fieldEl) fieldEl.classList.add("is-invalid");
									if (errorEl) errorEl.textContent = result.errors[field];
								}
							}
						}
					}
				} catch (e) {
					showError("Invalid response from server");
					console.error("JSON parse error:", e);
				}
			} else {
				if (xhr.status === 413) {
					showError(
						"File too large. Please ensure your video is less than 50MB."
					);
				} else {
					showError("Request failed. Try again. Status: " + xhr.status);
				}
			}

			getEl("addWorkoutBtn").disabled = false;
			getEl("submitBtn").disabled = false;
		};

		xhr.onerror = function () {
			closePopup();
			hideUploadStatus();
			showError("Request failed. Try again.");
			console.error("XHR error");

			getEl("addWorkoutBtn").disabled = false;
			getEl("submitBtn").disabled = false;
		};

		xhr.send(formData);
	});

	getEl("addWorkoutBtn").addEventListener("click", addSubWorkout);
});
