<?php
defined('BASEPATH') or exit('No direct script access allowed');

class AdminController extends CI_Controller
{

    public function __construct()
    {
        parent::__construct();
        $this->load->library('firebase');
        $this->load->library('session');
        $this->load->helper('url');
        $allowed_methods = ['login', 'postLogin'];
        if (!in_array($this->router->fetch_method(), $allowed_methods)) {
            $admin_session = $this->session->userdata('admin_session');
            if (empty($admin_session) || empty($admin_session['logged_in'])) {
                redirect('admin/login');
                exit;
            }
        }
    }
    public function index()
    {
        try {
            $firestore = $this->firebase->getFirestore();

            // Fetch 5 most recent orders
            $query = $firestore->collection('Orders')
                ->orderBy('created_at', 'DESC')
                ->limit(5);

            $documents = $query->documents();
            $recent_orders = [];

            foreach ($documents as $doc) {
                if ($doc->exists()) {
                    $data = $doc->data();
                    $data['id'] = $doc->id();
                    $recent_orders[] = $data;
                }
            }


            $this->load->view('admin/layouts/app', [
                'content' => $this->load->view('admin/landing_page', [
                    'recent_orders' => $recent_orders
                ], true)
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error fetching recent orders: ' . $e->getMessage());
            $this->load->view('admin/layouts/app', [
                'content' => $this->load->view('admin/landing_page', [
                    'recent_orders' => []
                ], true)
            ]);
        }
    }

    public function login()
    {
        return $this->load->view('admin/pages/login');
    }
    public function postLogin()
    {
        $this->output->set_content_type('application/json');

        try {
            $input = json_decode($this->input->raw_input_stream, true);

            $email = trim($input['email'] ?? '');
            $password = trim($input['password'] ?? '');

            if (empty($email) || empty($password)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Email and password are required.'
                ]);
                return;
            }

            // Get Firestore instance
            $firestore = $this->firebase->getFirestore();

            // Query both email and userType = admin
            $usersRef = $firestore->collection('Users');
            $query = $usersRef
                ->where('email', '=', $email)
                ->where('userType', '=', 'admin')
                ->limit(1);

            $documents = $query->documents();

            if ($documents->isEmpty()) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid email or password.'
                ]);
                return;
            }

            foreach ($documents as $doc) {
                if ($doc->exists()) {
                    $adminData = $doc->data();

                    //  If your password is plain
                    if ($adminData['password'] === $password) {
                        $this->session->set_userdata('admin_session', [
                            'adminId'   => $doc->id(),
                            'email'     => $adminData['email'],
                            'name'      => $adminData['name'] ?? 'Admin',
                            'logged_in' => true
                        ]);

                        echo json_encode([
                            'success'  => true,
                            'message'  => 'Login successful!',
                            'redirect' => base_url('admin/')
                        ]);
                        return;
                    }

                    // ✅ If you later hash passwords, use this instead:
                    // if (password_verify($password, $adminData['password'])) { ... }
                }
            }

            // If no match
            echo json_encode([
                'success' => false,
                'message' => 'Invalid email or password.'
            ]);
        } catch (Exception $e) {
            log_message('error', 'Login error: ' . $e->getMessage());
            echo json_encode([
                'success' => false,
                'message' => 'Server error. Please try again later.'
            ]);
        }
    }


    public function logout()
    {
        $this->session->unset_userdata('admin_session');
        $this->session->sess_destroy();
        redirect('admin/login');
    }
    public function accountPage()
    {
        $admin_session = $this->session->userdata('admin_session');

        // Check if logged in properly
        if (empty($admin_session) || empty($admin_session['logged_in'])) {
            redirect('admin/login');
            exit;
        }

        $data['admin_name']  = $admin_session['name'] ?? 'Admin';
        $data['admin_email'] = $admin_session['email'] ?? 'admin@example.com';
        $data['admin_role']  = 'admin';

        $this->load->view('admin/layouts/app', [
            'content' => $this->load->view('admin/pages/profile_page', $data, true)
        ]);
    }

    public function updatePassword()
    {
        $admin_session = $this->session->userdata('admin_session');

        if (empty($admin_session) || empty($admin_session['logged_in'])) {
            redirect('admin/login');
            exit;
        }

        $admin_id = $admin_session['adminId'];

        $old_password = $this->input->post('old_password');
        $new_password = $this->input->post('new_password');
        $confirm_password = $this->input->post('confirm_password');

        if ($new_password !== $confirm_password) {
            $this->session->set_flashdata('error', 'New passwords do not match.');
            redirect('admin/profile');
            return;
        }

        // Fetch admin from Firestore instead of MySQL
        $firestore = $this->firebase->getFirestore();
        $doc = $firestore->collection('Users')->document($admin_id)->snapshot();

        if (!$doc->exists()) {
            $this->session->set_flashdata('error', 'Admin account not found.');
            redirect('admin/profile');
            return;
        }

        $adminData = $doc->data();

        // Verify old password (plain or hashed)
        if ($adminData['password'] !== $old_password) {
            $this->session->set_flashdata('error', 'Old password is incorrect.');
            redirect('admin/profile');
            return;
        }

        // Update password in Firestore
        $firestore->collection('Users')->document($admin_id)->set([
            'password' => $new_password,
            'updated_at' => round(microtime(true) * 1000)
        ], ['merge' => true]);

        $this->session->set_flashdata('success', 'Password updated successfully.');
        redirect('admin/profile');
    }


    public function Orders()
    {
        try {
            $firestore = $this->firebase->getFirestore();

            // Fetch ALL orders (sorted by created_at DESC)
            $query = $firestore->collection('Orders')
                ->orderBy('created_at', 'DESC');

            $documents = $query->documents();
            $all_orders = [];

            foreach ($documents as $doc) {
                if ($doc->exists()) {
                    $data = $doc->data();
                    $data['id'] = $doc->id();
                    $all_orders[] = $data;
                }
            }

            // Load Orders page view and pass data
            $this->load->view('admin/layouts/app', [
                'content' => $this->load->view('admin/pages/orders', [
                    'orders' => $all_orders
                ], true)
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error fetching all orders: ' . $e->getMessage());

            // Load page even if query fails
            $this->load->view('admin/layouts/app', [
                'content' => $this->load->view('admin/pages/orders', [
                    'orders' => []
                ], true)
            ]);
        }
    }
    public function OrderDetails($orderId)
    {
        try {
            $firestore = $this->firebase->getFirestore();

            // 1️⃣ Get the order document
            $orderDoc = $firestore->collection('Orders')->document($orderId)->snapshot();
            if (!$orderDoc->exists()) {
                show_error('Order not found', 404);
            }

            $orderData = $orderDoc->data();
            $orderData['id'] = $orderDoc->id();

            // 2️⃣ Get the related customer data
            $customerId = $orderData['Customer_data_id'] ?? null;
            $customerData = [];
            if ($customerId) {
                $customerDoc = $firestore->collection('Customer_data')->document($customerId)->snapshot();
                if ($customerDoc->exists()) {
                    $customerData = $customerDoc->data();
                    $customerData['id'] = $customerDoc->id();
                }
            }

            // Debug: print both arrays
            // print_r($orderData);
            // print_r($customerData);
            // die;

            // 3️⃣ Load Order Details view
            $this->load->view('admin/layouts/app', [
                'content' => $this->load->view('admin/pages/order_details', [
                    'order' => $orderData,
                    'customer' => $customerData
                ], true)
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error fetching order details: ' . $e->getMessage());
            show_error('An error occurred while fetching order details.', 500);
        }
    }


    public function Products()
    {
        try {
            $firestore = $this->firebase->getFirestore();
            $query = $firestore->collection('products');
            $documents = $query->documents();

            $products = [];
            foreach ($documents as $doc) {
                if ($doc->exists()) {
                    $data = $doc->data();
                    $data['id'] = $doc->id();
                    $products[] = $data;
                }
            }

            // Pass products to your view
            $this->load->view('admin/layouts/app', [
                'content' => $this->load->view('admin/pages/products', [
                    'products' => $products
                ], true)
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error fetching products: ' . $e->getMessage());
            $this->load->view('admin/layouts/app', [
                'content' => $this->load->view('admin/pages/products', [
                    'products' => []
                ], true)
            ]);
        }
    }
    public function Customers()
    {
        $firestore = $this->firebase->getFirestore();

        // Pagination setup
        $limit = 10;
        $page = $this->input->get('page') ?? 1;
        $offset = ($page - 1) * $limit;

        // Fetch Users
        $usersRef = $firestore->collection('Users');
        $userDocs = $usersRef->documents();

        // Fetch Orders
        $ordersRef = $firestore->collection('Orders');
        $orderDocs = $ordersRef->documents();

        $customersData = [];

        // Build initial customer list exactly as in DB
        foreach ($userDocs as $userDoc) {
            if (!$userDoc->exists()) continue;

            $user = $userDoc->data();

            // Skip admin users
            if (isset($user['userType']) && strtolower($user['userType']) === 'admin') {
                continue;
            }

            // Convert isActive boolean to status string
            $isActive = $user['isActive'] ?? true;
            $status = $isActive ? 'active' : 'inactive';

            $customersData[$userDoc->id()] = [
                'id' => $userDoc->id(),
                'name' => $user['name'] ?? '',
                'email' => $user['email'] ?? '',
                'phone' => $user['phone'] ?? '',
                'address' => $user['address'] ?? '',
                'city' => $user['city'] ?? '',
                'postal_code' => $user['postal_code'] ?? '',
                'status' => $status, // Converted from isActive
                'isActive' => $isActive, // Original field
                'created_at' => $user['created_at'] ?? '',
                'orders' => [],
                'total_orders' => 0
            ];
        }

        // Attach Orders exactly as in DB
        foreach ($orderDocs as $orderDoc) {
            if (!$orderDoc->exists()) continue;

            $order = $orderDoc->data();
            $uid = $order['user_id'] ?? null;

            if (!$uid || !isset($customersData[$uid])) continue;

            $customersData[$uid]['orders'][] = [
                'id' => $orderDoc->id(),
                'customer_name' => $order['customer_name'] ?? '',
                'status' => $order['status'] ?? '',
                'images' => $order['images'] ?? [],
                'notes' => $order['notes'] ?? '',
                'phone' => $order['phone'] ?? '',
                'address' => $order['address'] ?? '',
                'city' => $order['city'] ?? '',
                'postal_code' => $order['postal_code'] ?? '',
                'created_at' => $order['created_at'] ?? ''
            ];

            $customersData[$uid]['total_orders']++;
        }

        // Convert to indexed array for view
        $customersData = array_values($customersData);

        // Pagination slice
        $totalCustomers = count($customersData);
        $customers = array_slice($customersData, $offset, $limit);

        $pagination = [
            'current_page' => (int)$page,
            'total_pages' => ceil($totalCustomers / $limit),
            'total_customers' => $totalCustomers
        ];

        // Load view
        $this->load->view(
            'admin/layouts/app',
            ['content' => $this->load->view('admin/pages/customers', [
                'customers' => $customers,
                'pagination' => $pagination,
                'all_customers' => $customersData // JavaScript ke liye
            ], true)]
        );
    }

    // New function for customer status update
    public function updateCustomerStatus()
    {
        try {
            $customerId = $this->input->post('customer_id');
            $status = $this->input->post('status');
            $reason = $this->input->post('reason');

            if (!$customerId || !$status) {
                throw new Exception('Customer ID and status are required');
            }

            $firestore = $this->firebase->getFirestore();
            $userRef = $firestore->collection('Users')->document($customerId);

            // Check if customer exists
            if (!$userRef->snapshot()->exists()) {
                throw new Exception('Customer not found');
            }

            // Convert status to isActive boolean
            $isActive = ($status === 'active') ? true : false;

            // Firebase update format - array of [fieldPath, value] pairs
            $updateData = [
                ['path' => 'isActive', 'value' => $isActive],
                ['path' => 'updated_at', 'value' => time() * 1000]
            ];

            // Add deactivation reason if provided and deactivating
            if ($reason && $status === 'inactive') {
                $updateData[] = ['path' => 'deactivation_reason', 'value' => $reason];
            }

            // Perform update - CORRECT FIREBASE FORMAT
            $userRef->update($updateData);

            echo json_encode([
                'success' => true,
                'message' => 'Customer status updated successfully'
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }



    // public function Customers()
    // {
    //     $firestore = $this->firebase->getFirestore();
    //     $usersRef = $firestore->collection('Users');

    //     $documents = $usersRef->documents();
    //     $customers = [];

    //     foreach ($documents as $doc) {
    //         if ($doc->exists()) {
    //             $data = $doc->data();

    //             // Skip admin users
    //             if (isset($data['userType']) && strtolower($data['userType']) === 'admin') {
    //                 continue;
    //             }

    //             // Format timestamps
    //             $createdAt = $lastActive = null;
    //             if (!empty($data['created_at'])) {
    //                 $createdAt = is_object($data['created_at']) && method_exists($data['created_at'], 'getTimestamp')
    //                     ? date('Y-m-d H:i:s', $data['created_at']->getTimestamp())
    //                     : date('Y-m-d H:i:s', $data['created_at'] / 1000);
    //             }
    //             if (!empty($data['last_active'])) {
    //                 $lastActive = is_object($data['last_active']) && method_exists($data['last_active'], 'getTimestamp')
    //                     ? date('Y-m-d H:i:s', $data['last_active']->getTimestamp())
    //                     : date('Y-m-d H:i:s', $data['last_active'] / 1000);
    //             }

    //             // Fetch Orders for this user
    //             $orders = [];
    //             $ordersRef = $usersRef->document($doc->id())->collection('Orders');
    //             foreach ($ordersRef->documents() as $orderDoc) {
    //                 if ($orderDoc->exists()) {
    //                     $orderData = $orderDoc->data();
    //                     $orders[] = [
    //                         'id' => $orderDoc->id(),
    //                         'date' => $orderData['date'] ?? '',
    //                         'amount' => $orderData['amount'] ?? 0,
    //                         'status' => $orderData['status'] ?? 'pending',
    //                         'items' => $orderData['items'] ?? 0,
    //                     ];
    //                 }
    //             }


    //             $customers[] = [
    //                 'id'           => $doc->id(),
    //                 'name'         => $data['name'] ?? 'N/A',
    //                 'email'        => $data['email'] ?? 'N/A',
    //                 'phone'        => $data['phone'] ?? 'N/A',
    //                 'status'       => !empty($data['isActive']) ? 'active' : 'inactive',
    //                 'total_orders' => count($orders),
    //                 'created_at'   => $createdAt,
    //                 'last_active'  => $lastActive,
    //                 'address'      => $data['address'] ?? '',
    //                 'city'         => $data['city'] ?? '',
    //                 'postal_code'  => $data['postal_code'] ?? '',
    //                 'orders'       => $orders, // include orders
    //             ];
    //         }
    //     }

    //     $this->load->view(
    //         'admin/layouts/app',
    //         ['content' => $this->load->view('admin/pages/customers', [
    //             'customers' => $customers
    //         ], true)]
    //     );
    // }

    public function addProduct()
    {
        $this->output->set_content_type('application/json');

        try {
            $firestore = $this->firebase->getFirestore();

            $productData = [
                'name' => $this->input->post('name'),
                'subtitle' => $this->input->post('subtitle'),
                'price' => (float) $this->input->post('price'),
                'description' => $this->input->post('description'),
                'image' => $this->input->post('image'),
                'thumbnails' => $this->input->post('thumbnails') ? json_decode($this->input->post('thumbnails'), true) : [],
                'status' => $this->input->post('status'),
                'created_at' => round(microtime(true) * 1000),
                'updated_at' => round(microtime(true) * 1000)
            ];

            $docRef = $firestore->collection('products')->add($productData);

            echo json_encode([
                'status' => 'success',
                'message' => 'Product added successfully',
                'product_id' => $docRef->id()
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error adding product: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to add product'
            ]);
        }
    }

    public function updateProduct($productId)
    {
        $this->output->set_content_type('application/json');

        try {
            $firestore = $this->firebase->getFirestore();

            $productData = [
                'name' => $this->input->post('name'),
                'subtitle' => $this->input->post('subtitle'),
                'price' => (float) $this->input->post('price'),
                'description' => $this->input->post('description'),
                'image' => $this->input->post('image'),
                'thumbnails' => $this->input->post('thumbnails') ? json_decode($this->input->post('thumbnails'), true) : [],
                'status' => $this->input->post('status'),
                'updated_at' => round(microtime(true) * 1000)
            ];

            $firestore->collection('products')->document($productId)->set($productData, ['merge' => true]);

            echo json_encode([
                'status' => 'success',
                'message' => 'Product updated successfully'
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error updating product: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update product'
            ]);
        }
    }

    public function deleteProduct($productId)
    {
        $this->output->set_content_type('application/json');

        try {
            $firestore = $this->firebase->getFirestore();
            $firestore->collection('products')->document($productId)->delete();

            echo json_encode([
                'status' => 'success',
                'message' => 'Product deleted successfully'
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error deleting product: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to delete product'
            ]);
        }
    }

    public function toggleProductStatus($productId)
    {
        $this->output->set_content_type('application/json');

        try {
            $firestore = $this->firebase->getFirestore();
            $productDoc = $firestore->collection('products')->document($productId)->snapshot();

            if (!$productDoc->exists()) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Product not found'
                ]);
                return;
            }

            $currentStatus = $productDoc->data()['status'];
            $newStatus = $currentStatus === 'active' ? 'inactive' : 'active';

            $firestore->collection('products')->document($productId)->set([
                'status' => $newStatus,
                'updated_at' => round(microtime(true) * 1000)
            ], ['merge' => true]);

            echo json_encode([
                'status' => 'success',
                'message' => 'Product status updated',
                'new_status' => $newStatus
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error toggling product status: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update product status'
            ]);
        }
    }

    public function uploadImage()
    {
        $this->output->set_content_type('application/json');

        try {
            $config['upload_path'] = './uploads/products/';
            $config['allowed_types'] = 'jpg|jpeg|png|gif|webp';
            $config['max_size'] = 2048; // 2MB
            $config['encrypt_name'] = true;

            if (!is_dir($config['upload_path'])) {
                mkdir($config['upload_path'], 0777, true);
            }

            $this->load->library('upload', $config);

            if (!$this->upload->do_upload('image')) {
                echo json_encode([
                    'status' => 'error',
                    'message' => $this->upload->display_errors()
                ]);
                return;
            }

            $upload_data = $this->upload->data();
            $image_url = base_url('uploads/products/' . $upload_data['file_name']);

            echo json_encode([
                'status' => 'success',
                'image_url' => $image_url,
                'file_name' => $upload_data['file_name']
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error uploading image: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to upload image'
            ]);
        }
    }
    public function test_add_orders()
    {
        // Set JSON header
        $this->output->set_content_type('application/json');

        // Sample orders to add
        $sample_orders = [
            [
                'customer_name' => 'Alice Johnson',
                'phone'         => '+92 300 1111111',
                'address'       => '123 Main Street, Block A',
                'city'          => 'Islamabad',
                'postal_code'   => '44000',
                'notes'         => 'Test order - Pending',
                'images'        => [], // Empty for test
                'status'        => 'pending',
            ],
            [
                'customer_name' => 'Bob Smith',
                'phone'         => '+92 301 2222222',
                'address'       => '456 Garden Ave, Block B',
                'city'          => 'Lahore',
                'postal_code'   => '54000',
                'notes'         => 'Test order - Rejected',
                'images'        => [],
                'status'        => 'rejected',
            ],
            [
                'customer_name' => 'Charlie Lee',
                'phone'         => '+92 302 3333333',
                'address'       => '789 Park Road, Block C',
                'city'          => 'Karachi',
                'postal_code'   => '75000',
                'notes'         => 'Test order - Accepted',
                'images'        => [],
                'status'        => 'accepted',
            ],
            [
                'customer_name' => 'Diana Prince',
                'phone'         => '+92 303 4444444',
                'address'       => '101 Ocean Drive, Block D',
                'city'          => 'Peshawar',
                'postal_code'   => '25000',
                'notes'         => 'Test order - Processing',
                'images'        => [],
                'status'        => 'processing',
            ],
            [
                'customer_name' => 'Ethan Hunt',
                'phone'         => '+92 304 5555555',
                'address'       => '202 Sunset Blvd, Block E',
                'city'          => 'Quetta',
                'postal_code'   => '87300',
                'notes'         => 'Test order - Completed',
                'images'        => [],
                'status'        => 'completed',
            ],
        ];

        $added_orders = [];
        $errors = [];

        foreach ($sample_orders as $order) {
            // Add timestamps
            $order['user_id']    = $this->session->userdata('session_data')['userId'] ?? null;
            $order['created_at'] = round(microtime(true) * 1000);
            $order['updated_at'] = round(microtime(true) * 1000);

            try {
                $docRef = $this->firebase->getFirestore()->collection('Orders')->add($order);
                $added_orders[] = $docRef->id();
            } catch (Exception $e) {
                $errors[] = $e->getMessage();
                log_message('error', 'Firestore test insert error: ' . $e->getMessage());
            }
        }

        // Return JSON response
        echo json_encode([
            'status' => empty($errors) ? 'success' : 'partial_error',
            'added_orders' => $added_orders,
            'errors' => $errors
        ]);
    }
}
