<?php
defined('BASEPATH') or exit('No direct script access allowed');

class OrderController extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->helper(['form', 'url']);
        $this->load->library(['session', 'form_validation', 'upload']);
        $this->load->library('firebase');
    }

    // Show upload form
    public function upload_pictures($Customer_data_id = null)
    {
        $userId = $this->session->userdata('user_id');
        $session_data = $this->session->flashdata();
        $this->load->view('layout/app', [
            'content' => $this->load->view('pages/upload_pictures', [
                'Customer_data_id' => $Customer_data_id,
                'userId' => $userId
            ], true)
        ]);
    }

    public function place_order()
    {
        // Set JSON header
        $this->output->set_content_type('application/json');

        // Debug: Log what we receive
        log_message('debug', 'FILES data: ' . print_r($_FILES, true));

        // Check if files exist
        if (!isset($_FILES['order_pictures']) || empty($_FILES['order_pictures']['name'][0])) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Please upload at least one image.'
            ]);
            return;
        }

        // Upload images
        $uploaded_images = $this->_upload_images();

        // Debug: Log upload result
        log_message('debug', 'Uploaded images: ' . print_r($uploaded_images, true));

        if (empty($uploaded_images)) {
            // Get upload errors
            $error_msg = $this->upload->display_errors('', '');
            log_message('error', 'Upload failed: ' . $error_msg);

            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to upload images. ' . ($error_msg ? $error_msg : 'Please ensure files are valid (jpg, jpeg, png, gif).')
            ]);
            return;
        }

        // Prepare order data
        $order_data = [
            'user_id'           => $this->session->userdata('user_id') ?? null,
            'Customer_data_id'  => $this->input->post('Customer_data_id') ?? null,
            'customer_name'  => $this->input->post('customer_name'),
            'phone'          => $this->input->post('phone'),
            'address'        => $this->input->post('address'),
            'city'           => $this->input->post('city'),
            'postal_code'    => $this->input->post('postal_code'),
            'notes'          => $this->input->post('notes'),
            'images'         => $uploaded_images,
            'status'         => 'pending',
            'created_at'     => round(microtime(true) * 1000),
            'updated_at'     => round(microtime(true) * 1000),
        ];

        // Save to Firestore
        try {
            $this->firebase->getFirestore()->collection('Orders')->add($order_data);

            echo json_encode([
                'status' => 'success',
                'message' => '🎉 Your order has been placed successfully!',
                'redirect' => base_url('order/success')
            ]);
        } catch (Exception $e) {
            log_message('error', 'Firestore error: ' . $e->getMessage());

            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to place order. Please try again later.'
            ]);
        }
    }

    /**
     * Upload multiple images
     */
    private function _upload_images()
    {
        $upload_path = './uploads/orders/';

        // Create directory if it doesn't exist
        if (!is_dir($upload_path)) {
            mkdir($upload_path, 0777, true);
        }

        $uploaded_files = [];
        $files_count = count($_FILES['order_pictures']['name']);

        for ($i = 0; $i < $files_count; $i++) {
            // Skip empty entries
            if (empty($_FILES['order_pictures']['name'][$i])) {
                continue;
            }

            // Check for upload errors
            if ($_FILES['order_pictures']['error'][$i] !== UPLOAD_ERR_OK) {
                log_message('error', 'File upload error code: ' . $_FILES['order_pictures']['error'][$i]);
                continue;
            }

            // Manually set up single file
            $_FILES['single_file']['name']     = $_FILES['order_pictures']['name'][$i];
            $_FILES['single_file']['type']     = $_FILES['order_pictures']['type'][$i];
            $_FILES['single_file']['tmp_name'] = $_FILES['order_pictures']['tmp_name'][$i];
            $_FILES['single_file']['error']    = $_FILES['order_pictures']['error'][$i];
            $_FILES['single_file']['size']     = $_FILES['order_pictures']['size'][$i];

            // Configure upload for this file
            $config['upload_path']   = $upload_path;
            $config['allowed_types'] = 'jpg|jpeg|png|gif';
            $config['max_size']      = 4096;
            $config['encrypt_name']  = TRUE; // Generate random name

            $this->upload->initialize($config);

            if ($this->upload->do_upload('single_file')) {
                $upload_data = $this->upload->data();
                $uploaded_files[] = base_url('uploads/orders/' . $upload_data['file_name']);
                log_message('debug', 'Successfully uploaded: ' . $upload_data['file_name']);
            } else {
                $error = $this->upload->display_errors('', '');
                log_message('error', 'Upload error for file ' . $i . ': ' . $error);
            }
        }

        return $uploaded_files;
    }

    /**
     * Success Page
     */
    public function order_success()
    {
        $this->load->view('layout/app', [
            'content' => $this->load->view('pages/order_success', [], true)
        ]);
    }

    /**
     * Update Order Status
     */
    public function update_status($orderId)
    {
        try {
            // Get JSON input
            $input = json_decode($this->input->raw_input_stream, true);

            if (!$input || empty($input['status'])) {
                throw new Exception('Invalid input data');
            }

            $firestore = $this->firebase->getFirestore();
            $orderRef = $firestore->collection('Orders')->document($orderId);

            // Check if order exists
            $snapshot = $orderRef->snapshot();
            if (!$snapshot->exists()) {
                throw new Exception('Order not found');
            }

            $updateData = [
                ['path' => 'status', 'value' => $input['status']],
                ['path' => 'updated_at', 'value' => round(microtime(true) * 1000)]
            ];

            // Add notes if provided
            if (!empty($input['notes'])) {
                $updateData[] = ['path' => 'notes', 'value' => $input['notes']];
            }

            $orderRef->update($updateData);

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Order status updated successfully'
                ]));
        } catch (Exception $e) {
            log_message('error', 'Update Order Status Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update status: ' . $e->getMessage()
                ]));
        }
    }
    /**
     * Accept Order
     */
    public function accept($orderId)
    {
        try {
            $firestore = $this->firebase->getFirestore();
            $orderRef = $firestore->collection('Orders')->document($orderId);

            // Check if order exists
            $snapshot = $orderRef->snapshot();
            if (!$snapshot->exists()) {
                throw new Exception('Order not found');
            }

            $orderRef->update([
                ['path' => 'status', 'value' => 'processing'],
                ['path' => 'updated_at', 'value' => round(microtime(true) * 1000)]
            ]);

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Order accepted and moved to processing'
                ]));
        } catch (Exception $e) {
            log_message('error', 'Accept Order Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to accept order: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Reject Order
     */
    public function reject($orderId)
    {
        try {
            // Get JSON input for reason
            $input = json_decode($this->input->raw_input_stream, true);
            $reason = isset($input['reason']) ? $input['reason'] : '';

            $firestore = $this->firebase->getFirestore();
            $orderRef = $firestore->collection('Orders')->document($orderId);

            // Check if order exists
            $snapshot = $orderRef->snapshot();
            if (!$snapshot->exists()) {
                throw new Exception('Order not found');
            }

            $updateData = [
                ['path' => 'status', 'value' => 'rejected'],
                ['path' => 'updated_at', 'value' => round(microtime(true) * 1000)]
            ];

            // Add rejection reason if provided
            if (!empty($reason)) {
                $updateData[] = ['path' => 'rejection_reason', 'value' => $reason];
            }

            $orderRef->update($updateData);

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Order rejected successfully'
                ]));
        } catch (Exception $e) {
            log_message('error', 'Reject Order Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to reject order: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Delete Order (Optional - if you want to add delete functionality)
     */
    public function delete($orderId)
    {
        try {
            $firestore = $this->firebase->getFirestore();
            $orderRef = $firestore->collection('Orders')->document($orderId);

            // Check if order exists
            $snapshot = $orderRef->snapshot();
            if (!$snapshot->exists()) {
                throw new Exception('Order not found');
            }

            $orderRef->delete();

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Order deleted successfully'
                ]));
        } catch (Exception $e) {
            log_message('error', 'Delete Order Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to delete order: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Get single order details (for AJAX)
     */
    public function get($orderId)
    {
        try {
            $firestore = $this->firebase->getFirestore();
            $orderRef = $firestore->collection('Orders')->document($orderId);
            $snapshot = $orderRef->snapshot();

            if (!$snapshot->exists()) {
                throw new Exception('Order not found');
            }

            $orderData = $snapshot->data();
            $orderData['id'] = $snapshot->id();

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'data' => $orderData
                ]));
        } catch (Exception $e) {
            log_message('error', 'Get Order Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to get order: ' . $e->getMessage()
                ]));
        }
    }
}
