<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Products extends CI_Controller
{

    public function __construct()
    {
        parent::__construct();
        $this->load->library(['session', 'form_validation', 'upload']);
        $this->load->helper(['url', 'form']);

        // Load Firebase library (make sure you have this library)
        $this->load->library('firebase');

        // Check admin authentication
        // $this->check_admin_auth();
    }

    private function check_admin_auth()
    {
        if (!$this->session->userdata('admin_logged_in')) {
            redirect('admin/login');
        }
    }

    /**
     * Main products page
     */
    public function index()
    {
        $firestore = $this->firebase->getFirestore();
        $products = [];

        // Fetch all product documents
        $documents = $firestore->collection('Products')->documents();

        foreach ($documents as $doc) {
            if ($doc->exists()) {
                $product = $doc->data();
                $product['id'] = $doc->id();
                $products[] = $product;
            }
        }



        // Pass data to nested view
        $content = $this->load->view('admin/pages/products', ['products' => $products], true);
        $this->load->view('admin/layouts/app', ['content' => $content]);
    }
    /**
     * Get all products from Firebase
     */
    public function get_all()
    {
        try {
            $firestore = $this->firebase->getFirestore();
            $productsRef = $firestore->collection('Products');
            $documents = $productsRef->documents();

            $products = [];
            foreach ($documents as $document) {
                if ($document->exists()) {
                    $productData = $document->data();
                    $productData['id'] = $document->id();
                    $products[] = $productData;
                }
            }

            // Sort by creation date or name
            usort($products, function ($a, $b) {
                $timeA = isset($a['created_at']) ? $a['created_at'] : 0;
                $timeB = isset($b['created_at']) ? $b['created_at'] : 0;
                return $timeB - $timeA; // Newest first
            });

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'data' => $products
                ]));
        } catch (Exception $e) {
            log_message('error', 'Firestore Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to fetch products: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Add new product to Firebase
     */
    public function add()
    {
        try {
            // Get JSON input
            $input = json_decode($this->input->raw_input_stream, true);

            if (!$input) {
                throw new Exception('Invalid input data');
            }

            // Validate required fields
            $required_fields = ['name', 'price', 'image', 'status'];
            foreach ($required_fields as $field) {
                if (empty($input[$field])) {
                    throw new Exception("Required field missing: $field");
                }
            }

            $firestore = $this->firebase->getFirestore();
            $productsRef = $firestore->collection('Products');

            $productData = [
                'name' => trim($input['name']),
                'subtitle' => isset($input['subtitle']) ? trim($input['subtitle']) : '',
                'price' => (float) $input['price'],
                'description' => isset($input['description']) ? trim($input['description']) : '',
                'image' => $input['image'],
                'thumbnails' => isset($input['thumbnails']) ? $input['thumbnails'] : [],
                'status' => $input['status'],
                'created_at' => round(microtime(true) * 1000),
                'updated_at' => round(microtime(true) * 1000)
            ];

            // Add document to Firestore
            $newDocRef = $productsRef->newDocument();
            $newDocRef->set($productData);

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Product added successfully',
                    'product_id' => $newDocRef->id()
                ]));
        } catch (Exception $e) {
            log_message('error', 'Add Product Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to add product: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Update existing product
     */
    public function update($productId)
    {
        try {
            // Get JSON input
            $input = json_decode($this->input->raw_input_stream, true);

            if (!$input) {
                throw new Exception('Invalid input data');
            }

            $firestore = $this->firebase->getFirestore();
            $productRef = $firestore->collection('Products')->document($productId);

            // Check if product exists
            $snapshot = $productRef->snapshot();
            if (!$snapshot->exists()) {
                throw new Exception('Product not found');
            }

            $updateData = [
                ['path' => 'name', 'value' => trim($input['name'])],
                ['path' => 'subtitle', 'value' => isset($input['subtitle']) ? trim($input['subtitle']) : ''],
                ['path' => 'price', 'value' => (float) $input['price']],
                ['path' => 'description', 'value' => isset($input['description']) ? trim($input['description']) : ''],
                ['path' => 'image', 'value' => $input['image']],
                ['path' => 'thumbnails', 'value' => isset($input['thumbnails']) ? $input['thumbnails'] : []],
                ['path' => 'status', 'value' => $input['status']],
                ['path' => 'updated_at', 'value' => round(microtime(true) * 1000)]
            ];

            $productRef->update($updateData);

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Product updated successfully'
                ]));
        } catch (Exception $e) {
            log_message('error', 'Update Product Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update product: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Delete product from Firebase
     */
    public function delete($productId)
    {
        try {
            $firestore = $this->firebase->getFirestore();
            $productRef = $firestore->collection('Products')->document($productId);

            // Check if product exists
            $snapshot = $productRef->snapshot();
            if (!$snapshot->exists()) {
                throw new Exception('Product not found');
            }

            $productRef->delete();

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Product deleted successfully'
                ]));
        } catch (Exception $e) {
            log_message('error', 'Delete Product Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to delete product: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Toggle product status (active/inactive)
     */
    public function toggle_status($productId)
    {
        try {
            $firestore = $this->firebase->getFirestore();
            $productRef = $firestore->collection('Products')->document($productId);

            // Check if product exists
            $snapshot = $productRef->snapshot();
            if (!$snapshot->exists()) {
                throw new Exception('Product not found');
            }

            $currentStatus = $snapshot->get('status');
            $newStatus = ($currentStatus === 'active') ? 'inactive' : 'active';

            $productRef->update([
                ['path' => 'status', 'value' => $newStatus],
                ['path' => 'updated_at', 'value' => round(microtime(true) * 1000)]
            ]);

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Product status updated to ' . $newStatus,
                    'new_status' => $newStatus
                ]));
        } catch (Exception $e) {
            log_message('error', 'Toggle Status Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to toggle status: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Upload image to server
     */
    public function upload_image()
    {
        try {
            $config['upload_path'] = './uploads/products/';
            $config['allowed_types'] = 'jpg|jpeg|png|gif|webp';
            $config['max_size'] = 2048; // 2MB
            $config['encrypt_name'] = TRUE;
            $config['remove_spaces'] = TRUE;

            // Create upload directory if it doesn't exist
            if (!is_dir($config['upload_path'])) {
                mkdir($config['upload_path'], 0777, TRUE);
            }

            $this->upload->initialize($config);

            if (!$this->upload->do_upload('image')) {
                $error = $this->upload->display_errors('', '');
                throw new Exception($error);
            }

            $upload_data = $this->upload->data();
            $image_url = base_url('uploads/products/' . $upload_data['file_name']);

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'message' => 'Image uploaded successfully',
                    'image_url' => $image_url,
                    'file_name' => $upload_data['file_name']
                ]));
        } catch (Exception $e) {
            log_message('error', 'Image Upload Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to upload image: ' . $e->getMessage()
                ]));
        }
    }

    /**
     * Get single product details
     */
    public function get($productId)
    {
        try {
            $firestore = $this->firebase->getFirestore();
            $productRef = $firestore->collection('Products')->document($productId);
            $snapshot = $productRef->snapshot();

            if (!$snapshot->exists()) {
                throw new Exception('Product not found');
            }

            $productData = $snapshot->data();
            $productData['id'] = $snapshot->id();

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'success',
                    'data' => $productData
                ]));
        } catch (Exception $e) {
            log_message('error', 'Get Product Error: ' . $e->getMessage());

            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode([
                    'status' => 'error',
                    'message' => 'Failed to get product: ' . $e->getMessage()
                ]));
        }
    }
}
