<!-- Main Content -->
<main>
    <div class="page-header">
        <h1 class="page-title">
            <i class="fas fa-shopping-cart me-2"></i>Orders Management
        </h1>
    </div>

    <!-- Stats Cards -->
    <?php
    $total = count($orders);
    $pending = count(array_filter($orders, fn($o) => ($o['status'] ?? '') === 'pending'));
    $processing = count(array_filter($orders, fn($o) => ($o['status'] ?? '') === 'processing'));
    $completed = count(array_filter($orders, fn($o) => ($o['status'] ?? '') === 'completed'));
    $rejected = count(array_filter($orders, fn($o) => ($o['status'] ?? '') === 'rejected'));
    ?>

    <div class="stats-row">
        <div class="stat-card">
            <div class="stat-icon all">
                <i class="fas fa-shopping-cart"></i>
            </div>
            <div class="stat-info">
                <h6>Total Orders</h6>
                <h3 id="total-orders"><?= $total ?></h3>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon pending">
                <i class="fas fa-clock"></i>
            </div>
            <div class="stat-info">
                <h6>Pending</h6>
                <h3 id="pending-orders"><?= $pending ?></h3>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon processing">
                <i class="fas fa-cog"></i>
            </div>
            <div class="stat-info">
                <h6>Processing</h6>
                <h3 id="processing-orders"><?= $processing ?></h3>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon completed">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="stat-info">
                <h6>Completed</h6>
                <h3 id="completed-orders"><?= $completed ?></h3>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon rejected">
                <i class="fas fa-times-circle"></i>
            </div>
            <div class="stat-info">
                <h6>Rejected</h6>
                <h3 id="rejected-orders"><?= $rejected ?></h3>
            </div>
        </div>
    </div>

    <!-- Orders Table Card -->
    <div class="datatable-card">
        <div class="datatable-header">
            <h3 class="datatable-title">All Orders</h3>
            <div class="header-actions">
                <div class="filter-buttons">
                    <button class="filter-btn active" data-status="all">All</button>
                    <button class="filter-btn" data-status="pending">Pending</button>
                    <button class="filter-btn" data-status="processing">Processing</button>
                    <button class="filter-btn" data-status="completed">Completed</button>
                    <button class="filter-btn" data-status="rejected">Rejected</button>
                </div>
                <div class="search-box">
                    <input type="text" id="searchInput" placeholder="Search orders..." class="form-control">
                </div>
            </div>
        </div>

        <!-- Table -->
        <div class="table-responsive">
            <table id="ordersTable" class="table table-hover" style="width:100%">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Order Details</th>
                        <th>Customer</th>
                        <th>Images</th>
                        <th>Status</th>
                        <th>Created</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody id="ordersTableBody">
                    <?php if (empty($orders)): ?>
                        <tr>
                            <td colspan="7" class="text-center text-muted py-4">No orders found.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($orders as $index => $order): ?>
                            <tr>
                                <td><?= $index + 1 ?></td>
                                <td>
                                    <strong>Order #<?= substr($order['id'] ?? 'N/A', 0, 8) ?></strong><br>
                                    <small class="text-muted"><?= htmlspecialchars($order['city'] ?? '') ?></small>
                                </td>
                                <td>
                                    <strong><?= htmlspecialchars($order['customer_name'] ?? 'N/A') ?></strong><br>
                                    <small class="text-muted"><?= htmlspecialchars($order['phone'] ?? '') ?></small>
                                </td>
                                <td>
                                    <?php if (!empty($order['images']) && is_array($order['images'])): ?>
                                        <div class="order-images">
                                            <img src="<?= $order['images'][0] ?>" alt="Order Image" class="order-image-thumb"
                                                onclick="viewOrder('<?= $order['id'] ?>')">
                                            <?php if (count($order['images']) > 1): ?>
                                                <div class="images-count">+<?= count($order['images']) - 1 ?></div>
                                            <?php endif; ?>
                                        </div>
                                    <?php else: ?>
                                        <span class="text-muted">No Images</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php
                                    $status = $order['status'] ?? 'pending';
                                    $statusClass = [
                                        'pending' => 'bg-warning',
                                        'processing' => 'bg-info',
                                        'completed' => 'bg-success',
                                        'rejected' => 'bg-danger'
                                    ][$status] ?? 'bg-secondary';
                                    ?>
                                    <span class="badge <?= $statusClass ?>"><?= ucfirst($status) ?></span>
                                </td>
                                <td>
                                    <?php
                                    if (!empty($order['created_at'])) {
                                        // Convert Firestore timestamp (ms) → seconds
                                        $timestamp = (int) ($order['created_at'] / 1000);
                                        echo date('d M Y, h:i A', $timestamp);
                                    } else {
                                        echo '—';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <div class="btn-group">
                                        <button class="btn btn-sm btn-outline-info" onclick="viewOrder('<?= $order['id'] ?>')" title="View">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button class="btn btn-sm btn-outline-primary" onclick="editOrderStatus('<?= $order['id'] ?>')" title="Edit Status">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <?php if ($status === 'pending'): ?>
                                            <button class="btn btn-sm btn-outline-success" onclick="updateOrderStatus('<?= $order['id'] ?>', 'processing')" title="Accept">
                                                <i class="fas fa-check"></i>
                                            </button>
                                            <button class="btn btn-sm btn-outline-danger" onclick="updateOrderStatus('<?= $order['id'] ?>', 'rejected')" title="Reject">
                                                <i class="fas fa-times"></i>
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <!-- Footer -->
        <div class="table-footer">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <div class="table-info" id="tableInfo">Showing <?= $total ?> orders</div>
                </div>
                <div class="col-md-6">
                    <nav>
                        <ul class="pagination justify-content-end" id="pagination">
                            <li class="page-item disabled"><a class="page-link">Prev</a></li>
                            <li class="page-item active"><a class="page-link">1</a></li>
                            <li class="page-item disabled"><a class="page-link">Next</a></li>
                        </ul>
                    </nav>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- View Order Modal -->
<div class="modal fade" id="viewOrderModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content shadow-lg border-0 rounded-3">
            <div class="modal-header bg-light">
                <h5 class="modal-title">
                    <i class="fas fa-receipt me-2 text-primary"></i> Order Details
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>

            <div class="modal-body">
                <div class="row">
                    <!-- Left column -->
                    <div class="col-md-6">
                        <div class="order-detail-row">
                            <div class="order-detail-label">Order ID:</div>
                            <div class="order-detail-value" id="modal-order-id"></div>
                        </div>
                        <div class="order-detail-row">
                            <div class="order-detail-label">Customer Name:</div>
                            <div class="order-detail-value" id="modal-customer-name"></div>
                        </div>
                        <div class="order-detail-row">
                            <div class="order-detail-label">Phone:</div>
                            <div class="order-detail-value" id="modal-phone"></div>
                        </div>
                        <div class="order-detail-row">
                            <div class="order-detail-label">Order Date:</div>
                            <div class="order-detail-value" id="modal-date"></div>
                        </div>
                    </div>

                    <!-- Right column -->
                    <div class="col-md-6">
                        <div class="order-detail-row">
                            <div class="order-detail-label">City:</div>
                            <div class="order-detail-value" id="modal-city"></div>
                        </div>
                        <div class="order-detail-row">
                            <div class="order-detail-label">Postal Code:</div>
                            <div class="order-detail-value" id="modal-postal"></div>
                        </div>
                        <div class="order-detail-row">
                            <div class="order-detail-label">Address:</div>
                            <div class="order-detail-value" id="modal-address"></div>
                        </div>
                        <div class="order-detail-row">
                            <div class="order-detail-label">Status:</div>
                            <div class="order-detail-value">
                                <span class="status-badge" id="modal-status-badge"></span>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="order-detail-row mt-3">
                    <div class="order-detail-label">Notes:</div>
                    <div class="order-detail-value" id="modal-notes"></div>
                </div>

                <hr class="my-4">

                <h6 class="mb-3"><i class="fas fa-images me-2 text-primary"></i>Order Images</h6>
                <div class="image-gallery" id="modal-images"></div>
            </div>

            <div class="modal-footer">
                <a href="#" id="view-full-order-btn" class="btn btn-primary">
                    <i class="fas fa-eye me-1"></i> View Full Details
                </a>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>

        </div>
    </div>
</div>


<!-- Edit Status Modal -->
<div class="modal fade" id="editStatusModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-edit me-2"></i>Update Order Status
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" id="edit-order-id">
                <div class="mb-3">
                    <label class="form-label">Order ID</label>
                    <input type="text" class="form-control" id="edit-order-display-id" readonly>
                </div>
                <div class="mb-3">
                    <label class="form-label">Customer Name</label>
                    <input type="text" class="form-control" id="edit-customer-name" readonly>
                </div>
                <div class="mb-3">
                    <label class="form-label">Current Status</label>
                    <input type="text" class="form-control" id="edit-current-status" readonly>
                </div>
                <div class="mb-3">
                    <label class="form-label">New Status *</label>
                    <select class="form-select" id="edit-new-status">
                        <option value="pending">Pending</option>
                        <option value="processing">Processing</option>
                        <option value="completed">Completed</option>
                        <option value="rejected">Rejected</option>
                    </select>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveStatusBtn">
                    <i class="fas fa-save me-1"></i>Update Status
                </button>
            </div>
        </div>
    </div>
</div>

<!-- jQuery -->
<script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<!-- Font Awesome -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">

<script>
    let ordersData = <?php echo json_encode($orders ?? []); ?>;
    let currentFilter = 'all';
    let currentPage = 1;
    const itemsPerPage = 10;
    let filteredData = [];

    // Initialize when document is ready
    $(document).ready(function() {
        initializeWithServerData();

        // Event Listeners
        $('#saveStatusBtn').on('click', saveStatusUpdate);
        $('#searchInput').on('input', filterOrders);

        $('.filter-btn').on('click', function() {
            $('.filter-btn').removeClass('active');
            $(this).addClass('active');
            currentFilter = $(this).data('status');
            filterOrders();
        });
    });
    const BASE_URL = "<?= base_url('admin/order-details/') ?>";

    // Example: dynamically set full order detail link
    function openOrderModal(order) {
        // Fill basic fields
        document.getElementById("modal-order-id").textContent = order.id || "-";
        document.getElementById("modal-customer-name").textContent = order.customer_name || "-";
        document.getElementById("modal-phone").textContent = order.phone || "-";
        document.getElementById("modal-date").textContent = new Date(order.created_at / 1000).toLocaleString();
        document.getElementById("modal-city").textContent = order.city || "-";
        document.getElementById("modal-postal").textContent = order.postal_code || "-";
        document.getElementById("modal-address").textContent = order.address || "-";
        document.getElementById("modal-notes").textContent = order.notes || "—";

        //  Set link dynamically
        const detailsBtn = document.getElementById("view-full-order-btn");
        console.log("Order ID:", order.id);
        console.log(detailsBtn);
        if (order.id) {
            detailsBtn.href = BASE_URL + order.id;
            detailsBtn.classList.remove("disabled");
        } else {
            detailsBtn.href = "#";
            detailsBtn.classList.add("disabled");
        }

        // Status badge
        const badge = document.getElementById("modal-status-badge");
        const status = (order.status || "pending").toLowerCase();
        badge.textContent = status.charAt(0).toUpperCase() + status.slice(1);
        badge.className = "status-badge " + (
            status === "accepted" || status === "completed" ?
            "success" :
            status === "rejected" ?
            "danger" :
            status === "pending" ?
            "warning" :
            "secondary"
        );

        // Show images
        const imgContainer = document.getElementById("modal-images");
        imgContainer.innerHTML = "";
        if (order.images && order.images.length) {
            order.images.forEach(url => {
                const img = document.createElement("img");
                img.src = url;
                img.alt = "Order image";
                imgContainer.appendChild(img);
            });
        }

        //  Show modal
        const modal = new bootstrap.Modal(document.getElementById('viewOrderModal'));
        modal.show();

        console.log("✅ Order ID set:", order.id, "→", detailsBtn.href);
    }


    // Initialize with server-side data
    function initializeWithServerData() {
        ordersData = ordersData.map((order, index) => ({
            ...order,
            serial: index + 1
        }));
        filteredData = [...ordersData];
        filterOrders();
    }

    // Filter and display orders
    function filterOrders() {
        const searchTerm = $('#searchInput').val().toLowerCase();
        filteredData = ordersData.filter(order => {
            const matchesSearch = !searchTerm ||
                (order.customer_name?.toLowerCase().includes(searchTerm)) ||
                (order.phone?.toLowerCase().includes(searchTerm)) ||
                (order.city?.toLowerCase().includes(searchTerm)) ||
                (order.id?.toLowerCase().includes(searchTerm));
            const matchesFilter = currentFilter === 'all' || order.status === currentFilter;
            return matchesSearch && matchesFilter;
        });
        renderTable();
        updatePagination();
        updateTableInfo();
        updateStats();
    }

    // Render table with data
    function renderTable() {
        const startIndex = (currentPage - 1) * itemsPerPage;
        const endIndex = startIndex + itemsPerPage;
        const pageData = filteredData.slice(startIndex, endIndex);
        const tbody = $('#ordersTableBody');

        tbody.empty();

        if (pageData.length === 0) {
            tbody.html(`
                <tr>
                    <td colspan="7" class="text-center py-4">
                        <i class="fas fa-shopping-cart fa-2x text-muted mb-3"></i>
                        <p class="text-muted">No orders found</p>
                    </td>
                </tr>
            `);
            return;
        }

        pageData.forEach(order => tbody.append(createTableRow(order)));
    }

    // Create table row HTML
    function createTableRow(order) {
        const createdDate = order.created_at ?
            new Date(parseInt(order.created_at)).toLocaleDateString('en-GB', {
                day: 'numeric',
                month: 'short',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            }) : '—';

        const statusClass = {
            'pending': 'bg-warning',
            'processing': 'bg-info',
            'completed': 'bg-success',
            'rejected': 'bg-danger'
        } [order.status] || 'bg-secondary';

        return `
            <tr>
                <td><strong>#${order.serial}</strong></td>
                <td>
                    <div class="order-info">
                        <div class="order-id">Order #${order.id ? order.id.substring(0, 8) : 'N/A'}</div>
                        <div class="order-city">${escapeHtml(order.city || '')}</div>
                    </div>
                </td>
                <td>
                    <div class="customer-info">
                        <div class="customer-name">${escapeHtml(order.customer_name || 'N/A')}</div>
                        <div class="customer-phone">${escapeHtml(order.phone || '')}</div>
                    </div>
                </td>
                <td>${createImagesHtml(order)}</td>
                <td><span class="badge ${statusClass}">${capitalizeFirst(order.status || 'pending')}</span></td>
                <td>${createdDate}</td>
                <td>${createActionsHtml(order)}</td>
            </tr>
        `;
    }

    // Create images HTML
    function createImagesHtml(order) {
        const images = order.images || [];
        if (images.length === 0) return '<span class="text-muted">No Images</span>';

        let html = '<div class="order-images">';
        html += `<img src="${images[0]}" alt="Order Image" class="order-image-thumb" onclick="viewOrder('${order.id}')">`;
        if (images.length > 1) html += `<div class="images-count">+${images.length - 1}</div>`;
        html += '</div>';
        return html;
    }

    // Create actions HTML
    function createActionsHtml(order) {
        let html = `
            <div class="btn-group btn-group-sm">
                <button class="btn btn-outline-info" onclick="viewOrder('${order.id}')" title="View">
                    <i class="fas fa-eye"></i>
                </button>
                <button class="btn btn-outline-primary" onclick="editOrderStatus('${order.id}')" title="Edit Status">
                    <i class="fas fa-edit"></i>
                </button>
        `;

        if (order.status === 'pending') {
            html += `
                <button class="btn btn-outline-success" onclick="confirmAcceptOrder('${order.id}')" title="Accept">
                    <i class="fas fa-check"></i>
                </button>
                <button class="btn btn-outline-danger" onclick="confirmRejectOrder('${order.id}')" title="Reject">
                    <i class="fas fa-times"></i>
                </button>
            `;
        }

        html += '</div>';
        return html;
    }

    // Update pagination
    function updatePagination() {
        const totalPages = Math.ceil(filteredData.length / itemsPerPage);
        const pagination = $('#pagination');
        pagination.empty();

        if (totalPages <= 1) {
            pagination.html('<li class="page-item disabled"><a class="page-link">1</a></li>');
            return;
        }

        // Previous button
        const prevDisabled = currentPage === 1 ? 'disabled' : '';
        pagination.append(`
            <li class="page-item ${prevDisabled}">
                <a class="page-link" href="#" onclick="changePage(${currentPage - 1}); return false;">Previous</a>
            </li>
        `);

        // Page numbers
        for (let i = 1; i <= totalPages; i++) {
            const active = i === currentPage ? 'active' : '';
            pagination.append(`
                <li class="page-item ${active}">
                    <a class="page-link" href="#" onclick="changePage(${i}); return false;">${i}</a>
                </li>
            `);
        }

        // Next button
        const nextDisabled = currentPage === totalPages ? 'disabled' : '';
        pagination.append(`
            <li class="page-item ${nextDisabled}">
                <a class="page-link" href="#" onclick="changePage(${currentPage + 1}); return false;">Next</a>
            </li>
        `);
    }

    // Change page
    function changePage(page) {
        currentPage = page;
        renderTable();
        updatePagination();
        updateTableInfo();
    }

    // Update table information
    function updateTableInfo() {
        const startIndex = (currentPage - 1) * itemsPerPage + 1;
        const endIndex = Math.min(currentPage * itemsPerPage, filteredData.length);
        $('#tableInfo').text(`Showing ${startIndex}-${endIndex} of ${filteredData.length} orders`);
    }

    // Update stats
    function updateStats() {
        const stats = {
            total: filteredData.length,
            pending: filteredData.filter(o => o.status === 'pending').length,
            processing: filteredData.filter(o => o.status === 'processing').length,
            completed: filteredData.filter(o => o.status === 'completed').length,
            rejected: filteredData.filter(o => o.status === 'rejected').length
        };

        $('#total-orders').text(stats.total);
        $('#pending-orders').text(stats.pending);
        $('#processing-orders').text(stats.processing);
        $('#completed-orders').text(stats.completed);
        $('#rejected-orders').text(stats.rejected);
    }

    // View Order Details
    window.viewOrder = function(orderId) {
        const order = ordersData.find(o => o.id === orderId);
        if (!order) {
            showToast('Error', 'Order not found', 'danger');
            return;
        }

        // Fill modal fields
        $('#modal-order-id').text(order.id || 'N/A');
        $('#modal-customer-name').text(order.customer_name || 'N/A');
        $('#modal-phone').text(order.phone || 'N/A');
        $('#modal-city').text(order.city || 'N/A');
        $('#modal-postal').text(order.postal_code || 'N/A');
        $('#modal-address').text(order.address || 'N/A');
        $('#modal-notes').text(order.notes || 'No notes provided');

        if (order.created_at) {
            const date = new Date(parseInt(order.created_at));
            $('#modal-date').text(date.toLocaleDateString('en-GB', {
                day: 'numeric',
                month: 'short',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            }));
        } else {
            $('#modal-date').text('N/A');
        }

        // Update "View Full Details" button dynamically
        const detailsBtn = document.getElementById("view-full-order-btn");
        if (detailsBtn && order.id) {
            detailsBtn.href = BASE_URL + order.id;
            detailsBtn.target = "_blank"; // Optional: open in new tab
            detailsBtn.classList.remove("disabled");
        }

        // Show images
        const imagesHtml = order.images?.length ?
            order.images.map(img => `<img src="${img}" class="gallery-image" onclick="window.open('${img}', '_blank')">`).join('') :
            '<p class="text-muted">No images available</p>';
        $('#modal-images').html(imagesHtml);

        // Show modal
        new bootstrap.Modal(document.getElementById('viewOrderModal')).show();
    };

    // Edit Order Status
    window.editOrderStatus = function(orderId) {
        const order = ordersData.find(o => o.id === orderId);
        if (!order) {
            showToast('Error', 'Order not found', 'danger');
            return;
        }

        $('#edit-order-id').val(order.id);
        $('#edit-order-display-id').val(order.id);
        $('#edit-customer-name').val(order.customer_name || 'N/A');
        $('#edit-current-status').val(capitalizeFirst(order.status || 'pending'));
        $('#edit-new-status').val(order.status || 'pending');

        new bootstrap.Modal(document.getElementById('editStatusModal')).show();
    };

    // Save Status Update
    async function saveStatusUpdate() {
        const orderId = $('#edit-order-id').val();
        const newStatus = $('#edit-new-status').val();

        try {
            const response = await fetch(`<?php echo site_url('admin/orders/update-status/'); ?>${orderId}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    status: newStatus
                })
            });

            const result = await response.json();
            if (result.status === 'success') {
                showToast('Success', result.message, 'success');

                // Update local data
                const orderIndex = ordersData.findIndex(o => o.id === orderId);
                if (orderIndex !== -1) {
                    ordersData[orderIndex].status = newStatus;
                    filterOrders();
                }

                bootstrap.Modal.getInstance(document.getElementById('editStatusModal')).hide();
            } else {
                throw new Error(result.message);
            }
        } catch (error) {
            console.error('Error updating status:', error);
            showToast('Error', error.message, 'danger');
        }
    }

    // ✅ NEW: Confirm Accept Order with SweetAlert
    window.confirmAcceptOrder = function(orderId) {
        const order = ordersData.find(o => o.id === orderId);
        if (!order) return;

        Swal.fire({
            title: 'Accept Order?',
            html: `
                <div class="text-start">
                    <p>Are you sure you want to accept this order?</p>
                    <div class="alert alert-info mt-2">
                        <strong>Order #${order.id.substring(0, 8)}</strong><br>
                        <small>Customer: ${order.customer_name}</small>
                    </div>
                </div>
            `,
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#1cc88a',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, Accept Order!',
            cancelButtonText: 'Cancel',
            reverseButtons: true,
            customClass: {
                confirmButton: 'btn btn-success',
                cancelButton: 'btn btn-secondary'
            },
            buttonsStyling: false
        }).then((result) => {
            if (result.isConfirmed) {
                updateOrderStatus(orderId, 'processing');
            }
        });
    };

    // ✅ NEW: Confirm Reject Order with SweetAlert
    window.confirmRejectOrder = function(orderId) {
        const order = ordersData.find(o => o.id === orderId);
        if (!order) return;

        Swal.fire({
            title: 'Reject Order?',
            html: `
                <div class="text-start">
                    <p>Are you sure you want to reject this order?</p>
                    <div class="alert alert-warning mt-2">
                        <strong>Order #${order.id.substring(0, 8)}</strong><br>
                        <small>Customer: ${order.customer_name}</small>
                    </div>
                    <div class="mt-3">
                        <label class="form-label">Reason for rejection (optional):</label>
                        <textarea id="rejection-reason" class="form-control" rows="3" placeholder="Enter reason for rejection..."></textarea>
                    </div>
                </div>
            `,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e74a3b',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, Reject Order!',
            cancelButtonText: 'Cancel',
            reverseButtons: true,
            customClass: {
                confirmButton: 'btn btn-danger',
                cancelButton: 'btn btn-secondary'
            },
            buttonsStyling: false,
            preConfirm: () => {
                const reason = document.getElementById('rejection-reason').value;
                return {
                    reason: reason
                };
            }
        }).then((result) => {
            if (result.isConfirmed) {
                const rejectionReason = result.value.reason;
                updateOrderStatus(orderId, 'rejected', rejectionReason);
            }
        });
    };

    // ✅ UPDATED: Quick status update with reason support
    async function updateOrderStatus(orderId, newStatus, reason = '') {
        try {
            const requestBody = {
                status: newStatus
            };

            // Add reason if provided
            if (reason) {
                requestBody.reason = reason;
            }

            const response = await fetch(`<?php echo site_url('admin/orders/update-status/'); ?>${orderId}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(requestBody)
            });

            const result = await response.json();
            if (result.status === 'success') {
                const actionText = newStatus === 'processing' ? 'accepted' : 'rejected';
                showToast('Success', `Order has been ${actionText} successfully`, 'success');

                // Update local data
                const orderIndex = ordersData.findIndex(o => o.id === orderId);
                if (orderIndex !== -1) {
                    ordersData[orderIndex].status = newStatus;
                    // Store rejection reason if provided
                    if (reason) {
                        ordersData[orderIndex].rejection_reason = reason;
                    }
                    filterOrders();
                }
            } else {
                throw new Error(result.message);
            }
        } catch (error) {
            console.error('Error updating status:', error);
            showToast('Error', error.message, 'danger');
        }
    }

    // Helper Functions
    function capitalizeFirst(str) {
        return str.charAt(0).toUpperCase() + str.slice(1);
    }

    function escapeHtml(unsafe) {
        if (!unsafe) return '';
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    // Toast Notification
    function showToast(title, message, type = 'info') {
        const colors = {
            success: '#1cc88a',
            danger: '#e74a3b',
            warning: '#f6c23e',
            info: '#4e73df'
        };
        const icons = {
            success: 'check-circle',
            danger: 'times-circle',
            warning: 'exclamation-circle',
            info: 'info-circle'
        };

        const toastHtml = `
            <div class="toast-notification" style="position: fixed; top: 20px; right: 20px; z-index: 9999; 
                        background: white; border-left: 4px solid ${colors[type]}; 
                        padding: 15px 20px; border-radius: 8px; 
                        box-shadow: 0 5px 20px rgba(0,0,0,0.2); min-width: 300px;
                        animation: slideIn 0.3s ease;">
                <div style="display: flex; align-items: center; gap: 12px;">
                    <i class="fas fa-${icons[type]}" style="color: ${colors[type]}; font-size: 20px;"></i>
                    <div>
                        <strong style="display: block; margin-bottom: 3px;">${title}</strong>
                        <span style="font-size: 14px; color: #666;">${message}</span>
                    </div>
                </div>
            </div>
        `;

        const $toast = $(toastHtml);
        $('body').append($toast);
        setTimeout(() => {
            $toast.css('animation', 'slideOut 0.3s ease');
            setTimeout(() => $toast.remove(), 300);
        }, 3000);
    }
</script>

<!-- Add SweetAlert2 CSS & JS -->
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.js"></script>

<style>
    .page-header {
        margin-bottom: 25px;
    }

    .page-title {
        color: #333;
        font-weight: 600;
        font-size: 1.8rem;
    }

    /* Stats Cards */
    .stats-row {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }

    .stat-card {
        background: white;
        border-radius: 10px;
        padding: 20px;
        box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        display: flex;
        align-items: center;
        transition: transform 0.3s;
    }

    .stat-card:hover {
        transform: translateY(-5px);
    }

    .stat-icon {
        width: 50px;
        height: 50px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 15px;
        color: white;
        font-size: 1.5rem;
    }

    .stat-icon.all {
        background: linear-gradient(135deg, var(--primary-color), #2e59d9);
    }

    .stat-icon.pending {
        background: linear-gradient(135deg, var(--warning-color), #e3b52d);
    }

    .stat-icon.processing {
        background: linear-gradient(135deg, var(--info-color), #2c9faf);
    }

    .stat-icon.completed {
        background: linear-gradient(135deg, var(--success-color), #17a673);
    }

    .stat-icon.rejected {
        background: linear-gradient(135deg, var(--danger-color), #c9302c);
    }

    .stat-info h6 {
        color: var(--secondary-color);
        font-size: 0.8rem;
        margin-bottom: 5px;
        text-transform: uppercase;
        font-weight: 700;
    }

    .stat-info h3 {
        color: #333;
        font-weight: 700;
        margin: 0;
    }

    /* DataTable Card */
    .datatable-card {
        background: white;
        border-radius: 10px;
        box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        overflow: hidden;
    }

    .datatable-header {
        padding: 20px 25px;
        border-bottom: 1px solid #e3e6f0;
        display: flex;
        justify-content: space-between;
        align-items: center;
        flex-wrap: wrap;
        gap: 15px;
    }

    .datatable-title {
        color: #333;
        font-weight: 600;
        margin: 0;
    }

    .header-actions {
        display: flex;
        align-items: center;
        gap: 15px;
        flex-wrap: wrap;
    }

    .filter-buttons {
        display: flex;
        background: #f8f9fc;
        border-radius: 8px;
        padding: 4px;
        flex-wrap: wrap;
    }

    .filter-btn {
        border: none;
        padding: 8px 16px;
        border-radius: 6px;
        background: transparent;
        font-size: 0.85rem;
        font-weight: 600;
        color: var(--secondary-color);
        transition: all 0.3s;
        cursor: pointer;
    }

    .filter-btn.active {
        background: white;
        color: var(--primary-color);
        box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    }

    .search-box {
        min-width: 250px;
    }

    .search-box input {
        border-radius: 8px;
    }

    /* Table Styles */
    .table-responsive {
        padding: 0 25px;
    }

    .table {
        margin-bottom: 0;
    }

    .table thead th {
        background: #f8f9fc;
        border-bottom: 2px solid #e3e6f0;
        font-size: 0.75rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        color: var(--secondary-color);
        font-weight: 700;
        padding: 15px 10px;
    }

    .table tbody td {
        padding: 15px 10px;
        vertical-align: middle;
        border-bottom: 1px solid #e3e6f0;
    }

    .table tbody tr:hover {
        background: #f8f9fc;
    }

    /* Order Images */
    .order-images {
        display: flex;
        gap: 5px;
        flex-wrap: wrap;
    }

    .order-image-thumb {
        width: 35px;
        height: 35px;
        border-radius: 6px;
        object-fit: cover;
        cursor: pointer;
        border: 2px solid #e3e6f0;
        transition: transform 0.3s;
    }

    .order-image-thumb:hover {
        transform: scale(1.2);
        border-color: var(--primary-color);
    }

    .images-count {
        width: 35px;
        height: 35px;
        border-radius: 6px;
        background: #f8f9fc;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 12px;
        color: var(--secondary-color);
        font-weight: 600;
    }

    /* Status Badges */
    .status-badge {
        padding: 6px 12px;
        border-radius: 50px;
        font-size: 12px;
        font-weight: 600;
        display: inline-block;
    }

    .status-badge.pending {
        background: #fff3cd;
        color: #856404;
    }

    .status-badge.processing {
        background: #cfe2ff;
        color: #084298;
    }

    .status-badge.completed {
        background: #d4edda;
        color: #155724;
    }

    .status-badge.rejected {
        background: #f8d7da;
        color: #721c24;
    }

    /* Table Footer */
    .table-footer {
        padding: 15px 25px;
        border-top: 1px solid #e3e6f0;
        background: #f8f9fc;
    }

    .table-info {
        font-size: 0.875rem;
        color: var(--secondary-color);
    }

    /* Pagination */
    .pagination {
        margin-bottom: 0;
    }

    .page-link {
        color: var(--primary-color);
        border: 1px solid #e3e6f0;
    }

    .page-item.active .page-link {
        background-color: var(--primary-color);
        border-color: var(--primary-color);
    }

    .page-link:hover {
        color: var(--primary-color);
        background-color: #e9ecef;
        border-color: #dee2e6;
    }

    /* Product Detail Rows (for modal) */
    .product-detail-row {
        display: flex;
        padding: 12px 0;
        border-bottom: 1px solid #f0f0f0;
    }

    .product-detail-row:last-child {
        border-bottom: none;
    }

    .product-detail-label {
        font-weight: 600;
        color: var(--secondary-color);
        width: 150px;
        font-size: 14px;
    }

    .product-detail-value {
        flex: 1;
        color: #333;
        font-size: 14px;
    }

    /* Image Gallery */
    .image-gallery {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
    }

    .gallery-image {
        width: 100px;
        height: 100px;
        object-fit: cover;
        border-radius: 8px;
        cursor: pointer;
        transition: transform 0.3s;
    }

    .gallery-image:hover {
        transform: scale(1.05);
    }

    /* Toast Animations */
    @keyframes slideIn {
        from {
            transform: translateX(400px);
            opacity: 0;
        }

        to {
            transform: translateX(0);
            opacity: 1;
        }
    }

    @keyframes slideOut {
        from {
            transform: translateX(0);
            opacity: 1;
        }

        to {
            transform: translateX(400px);
            opacity: 0;
        }
    }

    /* Responsive */
    @media (max-width: 768px) {
        .header-actions {
            flex-direction: column;
            align-items: stretch;
        }

        .search-box {
            min-width: auto;
        }

        .filter-buttons {
            order: -1;
        }

        .stats-row {
            grid-template-columns: 1fr;
        }
    }

    /* Order details layout */
    .order-detail-row {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        padding: 6px 0;
        border-bottom: 1px solid #f0f0f0;
    }

    .order-detail-label {
        flex: 0 0 35%;
        font-weight: 600;
        color: #555;
        font-size: 15px;
    }

    .order-detail-value {
        flex: 1;
        text-align: right;
        color: #222;
        font-size: 15px;
        word-break: break-word;
    }

    /* Notes section */
    #modal-notes {
        background: #f9f9f9;
        border-radius: 6px;
        padding: 10px 12px;
        color: #333;
        font-size: 14px;
    }

    /* Image gallery */
    .image-gallery {
        display: flex;
        flex-wrap: wrap;
        gap: 8px;
    }

    .image-gallery img {
        width: 90px;
        height: 90px;
        border-radius: 8px;
        border: 1px solid #ddd;
        object-fit: cover;
        transition: transform 0.2s ease;
        cursor: pointer;
    }

    .image-gallery img:hover {
        transform: scale(1.05);
        border-color: #007bff;
    }

    /* Status badge */
    .status-badge {
        display: inline-block;
        padding: 4px 10px;
        border-radius: 6px;
        font-size: 13px;
        font-weight: 600;
        color: white;
    }

    .status-badge.success {
        background-color: #28a745;
    }

    .status-badge.warning {
        background-color: #ffc107;
        color: #000;
    }

    .status-badge.danger {
        background-color: #dc3545;
    }

    .status-badge.secondary {
        background-color: #6c757d;
    }
</style>