document.addEventListener("DOMContentLoaded", function () {
	/* ============================================================
       STEP NAVIGATION + STATE
    ============================================================ */
	const steps = document.querySelectorAll(".form-step");
	const nextBtn = document.getElementById("nextBtn");
	const prevBtn = document.getElementById("prevBtn");
	const continueExistingBtn = document.getElementById("continueExistingBtn");
	const existingEmail = document.getElementById("existingEmail");
	let currentStep = 0;
	const formData = {};

	function showStep(index) {
		steps.forEach((step, i) => step.classList.toggle("active", i === index));
		window.scrollTo({ top: 0, behavior: "smooth" });
		nextBtn.textContent = index === steps.length - 1 ? "Submit" : "Next";
	}

	function nextStep() {
		if (!validateCurrentStep()) return;

		if (currentStep === 0) {
			const customerType = document.querySelector(
				'input[name="customer"]:checked'
			)?.value;
			currentStep = customerType === "existing" ? 1 : 2;
			return showStep(currentStep);
		}

		if (currentStep < steps.length - 1) {
			currentStep++;
			showStep(currentStep);
		} else {
			submitForm();
		}
	}

	function prevStep() {
		if (currentStep === 2) {
			const customerType = document.querySelector(
				'input[name="customer"]:checked'
			)?.value;
			currentStep = customerType === "new" ? 0 : 1;
		} else if (currentStep > 0) {
			currentStep--;
		}
		showStep(currentStep);
	}

	/* ============================================================
       VALIDATION
    ============================================================ */
	function validateCurrentStep() {
		const step = steps[currentStep];
		const errorDiv = step.querySelector(".error-message");
		if (errorDiv) errorDiv.classList.remove("show");

		let valid = true;
		let message = "";

		switch (currentStep) {
			case 0:
				valid = validateRadio(step, "customer");
				message = "Please select whether you are a new or existing customer.";
				break;
			case 1:
				valid = validateEmail(existingEmail.value.trim());
				if (!valid) existingEmail.classList.add("is-invalid");
				message = "Please enter a valid email.";
				break;
			default:
				valid = validateGeneric(step);
				message = "Please complete this step.";
		}

		if (!valid && errorDiv) {
			errorDiv.textContent = message;
			errorDiv.classList.add("show");
		}

		return valid;
	}

	function validateRadio(step, name) {
		return !!step.querySelector(`input[name="${name}"]:checked`);
	}

	function validateCheckbox(step, name) {
		return step.querySelectorAll(`input[name="${name}"]:checked`).length > 0;
	}

	function validateEmail(email) {
		return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
	}

	function validateGeneric(step) {
		const requiredInputs = step.querySelectorAll(
			"input[required], select[required]"
		);
		return Array.from(requiredInputs).every(
			(input) => input.value.trim() !== ""
		);
	}

	/* ============================================================
       FORM SUBMISSION
    ============================================================ */
	function submitForm() {
		const form = document.getElementById("multiStepForm");
		const formElements = form.querySelectorAll("input, select, textarea");

		formElements.forEach((el) => {
			if ((el.type === "radio" || el.type === "checkbox") && el.checked) {
				if (formData[el.name]) {
					if (Array.isArray(formData[el.name]))
						formData[el.name].push(el.value);
					else formData[el.name] = [formData[el.name], el.value];
				} else formData[el.name] = el.value;
			} else if (el.value) {
				formData[el.name || el.id] = el.value;
			}
		});

		fetch("/api/submit", {
			method: "POST",
			headers: { "Content-Type": "application/json" },
			body: JSON.stringify(formData),
		})
			.then((res) => res.json())
			.then(() => alert("Form submitted successfully!"))
			.catch(() => alert("Error submitting form."));
	}

	/* ============================================================
       EVENT HANDLERS
    ============================================================ */
	if (nextBtn) nextBtn.addEventListener("click", nextStep);
	if (prevBtn) prevBtn.addEventListener("click", prevStep);
	if (continueExistingBtn) {
		continueExistingBtn.addEventListener("click", function () {
			if (validateEmail(existingEmail.value.trim())) {
				nextStep();
			} else {
				existingEmail.classList.add("is-invalid");
			}
		});
	}

	showStep(currentStep);
});

//# sourceMappingURL=eligibility-form.js.map
document.addEventListener("DOMContentLoaded", function () {
	/* ============================================================
       HEIGHT & WEIGHT TOGGLES
    ============================================================ */
	const switchHeight = document.getElementById("switchHeight");
	if (switchHeight) {
		switchHeight.addEventListener("click", function () {
			const cmDiv = document.getElementById("heightCmDiv");
			const ftDiv = document.getElementById("heightFtDiv");
			cmDiv.classList.toggle("hidden");
			ftDiv.classList.toggle("hidden");
			this.textContent = cmDiv.classList.contains("hidden")
				? "Switch to centimeters"
				: "Switch to feet/inches";
		});
	}

	const switchWeight = document.getElementById("switchWeight");
	if (switchWeight) {
		switchWeight.addEventListener("click", function () {
			const kgDiv = document.getElementById("weightKgDiv");
			const stDiv = document.getElementById("weightStDiv");
			kgDiv.classList.toggle("hidden");
			stDiv.classList.toggle("hidden");
			this.textContent = kgDiv.classList.contains("hidden")
				? "Switch to kilograms"
				: "Switch to stones/pounds";
		});
	}

	/* ============================================================
       OPTION CARD SELECTION
    ============================================================ */
	document.querySelectorAll(".option-card").forEach((card) => {
		card.addEventListener("click", function () {
			document
				.querySelectorAll(".option-card")
				.forEach((c) => c.classList.remove("active"));
			this.classList.add("active");
			const value = this.getAttribute("data-value");
			const hiddenInput = document.getElementById("treatment_type");
			if (hiddenInput) hiddenInput.value = value;
		});
	});

	/* ============================================================
       "NONE OF THE ABOVE" CHECKBOXES
    ============================================================ */
	function setupNoneCheckbox(noneId, groupName) {
		const noneBox = document.getElementById(noneId);
		if (!noneBox) return;

		const others = document.querySelectorAll(
			`input[name="${groupName}"]:not(#${noneId})`
		);
		noneBox.addEventListener("change", function () {
			if (this.checked) {
				others.forEach((cb) => {
					cb.checked = false;
					cb.closest(".option-box")?.classList.remove("selected");
				});
			}
		});

		others.forEach((cb) => {
			cb.addEventListener("change", function () {
				if (this.checked) {
					noneBox.checked = false;
					noneBox.closest(".option-box")?.classList.remove("selected");
				}
			});
		});
	}

	setupNoneCheckbox("noneCondition", "conditions[]");
	setupNoneCheckbox("noneAllergy", "allergy[]");
	setupNoneCheckbox("noneMedical", "medical_history[]");
	setupNoneCheckbox("noneOtherHealth", "other_health_issues[]");

	/* ============================================================
       OPTION BOX INTERACTIONS
    ============================================================ */
	document.querySelectorAll(".option-box").forEach((box) => {
		box.addEventListener("click", function (e) {
			const input = this.querySelector("input");
			if (!input) return;

			if (input.type === "radio") {
				document
					.querySelectorAll(`input[name="${input.name}"]`)
					.forEach((r) =>
						r.closest(".option-box")?.classList.remove("selected")
					);
				input.checked = true;
				this.classList.add("selected");
			} else if (input.type === "checkbox") {
				input.checked = !input.checked;
				this.classList.toggle("selected", input.checked);
			}
		});
	});

	/* ============================================================
       CLEAR ERROR STYLES ON INPUT
    ============================================================ */
	document.querySelectorAll("input, select, textarea").forEach((el) => {
		el.addEventListener("input", () => el.classList.remove("is-invalid"));
	});
});
